package ru.yandex.solomon.core.conf.aggr;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValues;
import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Stepan Koltsov
 */
public class LabelValueExpr extends DefaultObject {

    private final Node[] nodes;

    public LabelValueExpr(Node[] nodes) {
        if (nodes.length == 0) {
            throw new RuntimeException("empty value name pairs");
        }
        this.nodes = nodes;
    }

    @Nullable
    public String eval(Labels labels, Labels optLabels) {
        if (nodes.length == 1) {
            return nodes[0].eval(labels, optLabels);
        }

        StringBuilder result = new StringBuilder();
        for (var node : nodes) {
            String part = node.eval(labels, optLabels);
            if (part == null) {
                return null;
            }
            result.append(part);
        }
        return result.toString();
    }

    public LabelValueExpr bind(Map<String, String> partial) {
        Node[] bind = new Node[nodes.length];
        for (int index = 0; index < nodes.length; index++) {
            bind[index] = nodes[index].bind(partial);
        }

        return new LabelValueExpr(mergeNeighborsAtom(bind));
    }

    public Set<String> getSubstitutionLabelNames() {
        Set<String> result = new HashSet<>();
        for (var node : nodes) {
            if (node instanceof NodeReplace) {
                result.add(((NodeReplace) node).labelName);
            }
        }
        return result;
    }

    public boolean isHostRequired() {
        for (var node : nodes) {
            if (node instanceof NodeReplace) {
                var labelName = ((NodeReplace) node).labelName;
                if (LabelKeys.HOST.equals(labelName)) {
                    return true;
                }
            }
        }

        return false;
    }

    private Node[] mergeNeighborsAtom(Node[] nodes) {
        if (nodes.length <= 1) {
            return nodes;
        }

        @Nullable
        NodeAtom prev = null;
        List<Node> result = new ArrayList<>(nodes.length);
        for (var node : nodes) {
            if (prev == null) {
                if (node instanceof NodeAtom) {
                    prev = (NodeAtom) node;
                } else {
                    result.add(node);
                }
            } else {
                if (node instanceof NodeAtom) {
                    prev = prev.merge((NodeAtom) node);
                } else {
                    result.add(prev);
                    result.add(node);
                    prev = null;
                }
            }
        }

        if (prev != null) {
            result.add(prev);
        }

        return result.toArray(new Node[0]);
    }

    private static class Builder {
        private final List<Node> nodes = new ArrayList<>(1);

        public LabelValueExpr build() {
            return new LabelValueExpr(nodes.toArray(new Node[0]));
        }

        public void appendAtom(String value) {
            nodes.add(new NodeAtom(value));
        }

        public void appendReplace(String labelName) {
            if (Strings.isNullOrEmpty(labelName)) {
                throw new RuntimeException("empty name");
            }

            nodes.add(new NodeReplace(labelName));
        }
    }

    public static LabelValueExpr parse(String expr) {
        Builder builder = new Builder();
        ParserSupport parser = new ParserSupport(expr);
        while (!parser.lookingAtEof()) {
            String value = parser.consumeUntil("{{");
            if (!Strings.isNullOrEmpty(value)) {
                builder.appendAtom(value);
            }
            if (parser.lookingAtEof()) {
                break;
            }
            parser.consume("{{");
            String labelName = parser.consumeUntil("}}");
            parser.consume("}}");
            builder.appendReplace(labelName);
        }

        return builder.build();
    }

    @Override
    public String toString() {
        return Joiner.on("").join(nodes);
    }

    private static boolean isValidExprChars(String expr) {
        String exprWithoutGlob = StringUtils.remove(StringUtils.remove(expr, '*'), '?');
        if (exprWithoutGlob.isEmpty()) {
            return true;
        }

        return StrictValidator.SELF.isValueValid(exprWithoutGlob);
    }

    private static boolean isValidExpr(String expr) {
        ParserSupport parser = new ParserSupport(expr);
        StringBuilder value = new StringBuilder();

        while (parser.hasNext()) {
            String valuePart = parser.consumeUntil("{{");
            value.append(valuePart);

            if (parser.lookingAtEof()) {
                break;
            }
            parser.consume("{{");

            String labelName = parser.consumeUntil("}}");
            if (!StrictValidator.SELF.isKeyValid(labelName)) {
                return false;
            }

            if (!parser.lookaheadIs("}}")) {
                return false;
            }
            parser.consume("}}");
        }

        return isValidExprChars(value.toString());
    }

    public static boolean isValid(String expr) {
        ParserSupport parser = new ParserSupport(expr);

        if (parser.lookaheadIs('!')) {
            parser.consume('!');
        }

        while (parser.hasNext()) {
            String value = parser.consumeUntil('|');
            if (value.isEmpty()) {
                return false;
            } else if (value.contains("{{")) {
                if (!isValidExpr(value)) {
                    return false;
                }
            } else if (!isValidExprChars(value) && !LabelValues.ABSENT.equals(value)) {
                return false;
            }

            if (parser.lookaheadIs('|')) {
                parser.consumeChar();

                if (parser.lookingAtEof()) {
                    return false;
                }
            }
        }
        return true;
    }

    private interface Node {
        String eval(Labels labels, Labels optLabels);
        Node bind(Map<String, String> partial);
    }

    private static class NodeAtom extends DefaultObject implements Node {
        private final String value;

        public NodeAtom(String value) {
            this.value = value;
        }

        @Override
        public String eval(Labels labels, Labels optLabels) {
            return value;
        }

        @Override
        public NodeAtom bind(Map<String, String> partial) {
            return this;
        }

        public NodeAtom merge(NodeAtom other) {
            return new NodeAtom(value + other.value);
        }

        @Override
        public String toString() {
            return value;
        }
    }

    private static class NodeReplace extends DefaultObject implements Node {
        private final String labelName;

        public NodeReplace(String labelName) {
            this.labelName = labelName;
        }

        @Override
        public String eval(Labels labels, Labels optLabels) {
            var label = labels.findByKey(labelName);
            if (label == null) {
                label = optLabels.findByKey(labelName);
            }

            return label != null ? label.getValue() : null;
        }

        @Override
        public Node bind(Map<String, String> partial) {
            var value = partial.get(labelName);
            if (value == null) {
                return this;
            } else {
                return new NodeAtom(value);
            }
        }

        @Override
        public String toString() {
            return "{{" + labelName + "}}";
        }
    }


}
