package ru.yandex.solomon.core.conf.aggr;

import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.labels.LabelValues;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class TargetParsed {

    @Nonnull
    private final String name;
    @Nonnull
    private final LabelValueExpr expr;
    private final boolean allowedToAddNewLabel;

    public TargetParsed(String name, LabelValueExpr value, boolean allowedToAddNewLabel) {
        this.name = name;
        this.expr = value;
        this.allowedToAddNewLabel = allowedToAddNewLabel;
    }

    @Nonnull
    public String getName() {
        return name;
    }

    @Nonnull
    public LabelValueExpr getExpr() {
        return expr;
    }

    public Set<String> getSubstitutionLabelNames() {
        return expr.getSubstitutionLabelNames();
    }

    public boolean isHostRequired() {
        return expr.isHostRequired();
    }

    public TargetParsed resolveShardKeyLabels(ShardKey shardKey) {
        return new TargetParsed(name, expr.bind(shardKey.toMap()), allowedToAddNewLabel);
    }

    public boolean apply(LabelsBuilder builder, Labels labels, Labels optLabels) {
        String valueToPut = expr.eval(labels, optLabels);

        // do not apply rule if failed to eval expr,
        // e. g. if there's no DC in expr `host={{DC}}`
        if (valueToPut == null) {
            return false;
        } else if (LabelValues.ABSENT.equals(valueToPut)) {
            builder.remove(name);
            return true;
        }

        if (!allowedToAddNewLabel && !builder.hasKey(name)) {
            // do not apply rule if source labels have no label with given name
            return false;
        }
        builder.add(name, valueToPut);
        return true;
    }

    @Nullable
    private static TargetParsed parse(String string, Set<String> keysOfAbsentSelectors) {
        ParserSupport parser = new ParserSupport(string);
        String name = parser.nextStringUpToSep('=');
        parser.consume('=');
        if (parser.lookingAtEof()) {
            // deprecated: delete label
            return null;
        } else {
            return new TargetParsed(name, LabelValueExpr.parse(parser.consumeRest()), keysOfAbsentSelectors.contains(name));
        }
    }

    @Nullable
    public static TargetParsed parseReplaceSpecials(String string, ShardKey shardKey, Set<String> keysOfAbsentSelectors) {
        TargetParsed r = parse(string, keysOfAbsentSelectors);
        return r != null ? r.resolveShardKeyLabels(shardKey) : null;
    }
}
