package ru.yandex.solomon.core.conf.flags;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.core.db.dao.ClusterFlagsDao;
import ru.yandex.solomon.core.db.dao.ProjectFlagsDao;
import ru.yandex.solomon.core.db.dao.ServiceFlagsDao;
import ru.yandex.solomon.core.db.dao.ShardFlagsDao;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.staffOnly.RedirectException;
import ru.yandex.solomon.staffOnly.RootLink;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
public class FeatureFlagsController {
    private static final Logger logger = LoggerFactory.getLogger(FeatureFlagsController.class);
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final FeatureFlagsHolderImpl holder;
    private final ProjectFlagsDao projectDao;
    private final ClusterFlagsDao clusterDao;
    private final ServiceFlagsDao serviceDao;
    private final ShardFlagsDao shardDao;

    @Autowired
    public FeatureFlagsController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            FeatureFlagsHolderImpl holder,
            ProjectFlagsDao projectDao,
            ClusterFlagsDao clusterDao,
            ServiceFlagsDao serviceDao,
            ShardFlagsDao shardDao
    ) {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.holder = holder;
        this.projectDao = projectDao;
        this.clusterDao = clusterDao;
        this.serviceDao = serviceDao;
        this.shardDao = shardDao;
    }

    @Bean
    public RootLink featureFlagsLinks() {
        return new RootLink("/featureFlags", "Feature flags");
    }

    @RequestMapping(value = "/featureFlags", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> featureFlags(ServerHttpRequest request, @RequestParam Map<String, String> params) {
        return authorize(request)
                .thenApply(ignore -> new FeatureFlagsPage("/featureFlags", params, holder).genString());
    }

    @RequestMapping("/featureFlags/Enable")
    public CompletableFuture<String> enableFlag(
            @RequestParam(value = "projectId", defaultValue = "") String projectId,
            @RequestParam(value = "clusterId", defaultValue = "") String clusterId,
            @RequestParam(value = "serviceId", defaultValue = "") String serviceId,
            @RequestParam(value = "shardId", defaultValue = "") String shardId,
            @RequestParam(value = "serviceProvider", defaultValue = "") String serviceProvider,
            @RequestParam(value = "flag") FeatureFlag flag,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenCompose(account -> updateFlag(projectId, clusterId, serviceId, shardId, serviceProvider, flag, true))
                .thenApply(ignore -> redirectBack(request))
                .whenComplete((aVoid, throwable) -> {
                    if (throwable != null) {
                        logger.error("/featureFlags/Enable failed", throwable);
                    }
                });
    }

    @RequestMapping("/featureFlags/Disable")
    public CompletableFuture<String> disableFlag(
            @RequestParam(value = "projectId", defaultValue = "") String projectId,
            @RequestParam(value = "clusterId", defaultValue = "") String clusterId,
            @RequestParam(value = "serviceId", defaultValue = "") String serviceId,
            @RequestParam(value = "shardId", defaultValue = "") String shardId,
            @RequestParam(value = "serviceProvider", defaultValue = "") String serviceProvider,
            @RequestParam(value = "flag") FeatureFlag flag,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenCompose(account -> updateFlag(projectId, clusterId, serviceId, shardId, serviceProvider, flag, false))
                .thenApply(ignore -> redirectBack(request))
                .whenComplete((aVoid, throwable) -> {
                    if (throwable != null) {
                        logger.error("/featureFlags/Disable failed", throwable);
                    }
                });
    }

    @RequestMapping("/featureFlags/Delete")
    public CompletableFuture<String> deleteFlag(
            @RequestParam(value = "projectId", defaultValue = "") String projectId,
            @RequestParam(value = "clusterId", defaultValue = "") String clusterId,
            @RequestParam(value = "serviceId", defaultValue = "") String serviceId,
            @RequestParam(value = "shardId", defaultValue = "") String shardId,
            @RequestParam(value = "serviceProvider", defaultValue = "") String serviceProvider,
            @RequestParam(value = "flag") FeatureFlag flag,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenCompose(account -> deleteFlag(projectId, clusterId, serviceId, shardId, serviceProvider, flag))
                .thenApply(ignore -> redirectBack(request))
                .whenComplete((aVoid, throwable) -> {
                    if (throwable != null) {
                        logger.error("/featureFlags/Delete failed", throwable);
                    }
                });
    }

    private CompletableFuture<Account> authorize(ServerHttpRequest request) {
        return authenticator.authenticate(request)
                .thenCompose(authorizer::authorize);
    }

    private CompletableFuture<?> updateFlag(String projectId, String clusterId, String serviceId, String shardId, String serviceProvider, FeatureFlag flag, boolean state) {
        requireNonNull(projectId);
        requireNonNull(clusterId);
        requireNonNull(serviceId);
        requireNonNull(shardId);
        requireNonNull(serviceProvider);
        requireNonNull(flag);
        var config = requireNonNull(holder.config);
        var op = new FeatureFlagsOperationUpdate(config, projectDao, clusterDao, serviceDao, shardDao, flag, state);
        return op.apply(projectId, clusterId, serviceId, shardId, serviceProvider);
    }

    private CompletableFuture<?> deleteFlag(String projectId, String clusterId, String serviceId, String shardId, String serviceProvider, FeatureFlag flag) {
        requireNonNull(projectId);
        requireNonNull(clusterId);
        requireNonNull(serviceId);
        requireNonNull(shardId);
        requireNonNull(serviceProvider);
        requireNonNull(flag);
        var config = requireNonNull(holder.config);
        var op = new FeatureFlagsOperationDelete(config, projectDao, clusterDao, serviceDao, shardDao, flag);
        return op.apply(projectId, clusterId, serviceId, shardId, serviceProvider);
    }

    private String redirectBack(ServerHttpRequest request) {
        String referer = request.getHeaders().getFirst("Referer");
        if (Strings.isNullOrEmpty(referer)) {
            throw new RedirectException("/featureFlags");
        }

        throw new RedirectException(referer);
    }
}
