package ru.yandex.solomon.core.conf.flags;

import java.util.EnumMap;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.Tasks;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.core.conf.ConfigNotInitialized;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfListener;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlags;
import ru.yandex.solomon.flags.FeatureFlagsConfig;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.flags.FeatureFlagsListener;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;
import ru.yandex.solomon.util.file.FileStorage;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class FeatureFlagsHolderImpl implements SolomonConfListener, FeatureFlagsListener, FeatureFlagsHolder {
    private static final Logger logger = LoggerFactory.getLogger(FeatureFlagsHolderImpl.class);
    private static final String FEATURE_FLAGS_STATE_FILE = "featureFlags.state";

    @Nullable
    volatile SolomonConfWithContext config;
    @Nullable
    volatile FeatureFlagsConfig flagsConfig;
    @Nullable
    volatile Int2ObjectMap<FeatureFlags> flagsByNumId;
    private final FileStorage storage;
    private final Metrics metrics;
    private final Tasks tasks = new Tasks();
    private final ObjectMapper mapper = new ObjectMapper();

    public FeatureFlagsHolderImpl(FileStorage storage, MetricRegistry registry) {
        this.storage = storage;
        this.metrics = new Metrics(registry);
        this.flagsConfig = loadStateFromFile(FEATURE_FLAGS_STATE_FILE, FeatureFlagsConfig.class);
    }

    @Override
    public void onConfigurationLoad(SolomonConfWithContext conf) {
        this.config = requireNonNull(conf);
        updateSnapshot();
    }

    @Override
    public void onConfigurationLoad(FeatureFlagsConfig flags) {
        this.flagsConfig = requireNonNull(flags);
        updateSnapshot();
        saveStateToFile(FEATURE_FLAGS_STATE_FILE, flags);
    }

    @Override
    public FeatureFlags flags(String projectId) {
        var snapshot = flagsConfig;
        if (snapshot == null) {
            throw new ConfigNotInitialized();
        }

        return snapshot.getFlags(projectId);
    }

    @Override
    public FeatureFlags flags(int numId) {
        var snapshot = ensureConfigLoaded();
        var flags = snapshot.get(numId);
        if (flags != null) {
            return flags;
        }

        var shard = requireNonNull(config).getShardByNumId(numId).getConfOrThrow();
        return flags(shard);
    }

    @Override
    public FeatureFlags flags(String project, String cluster, String service) {
        var snapshot = ensureConfigLoaded();
        var conf = requireNonNull(config);
        var shard = conf.findShardOrNull(ShardKey.create(project, cluster, service));
        if (shard != null) {
            var flags = snapshot.get(shard.getNumId());
            if (flags != null) {
                return flags;
            }
            return flags(shard);
        }

        return flags(project, service);
    }

    @Override
    public String define(FeatureFlag flag, String project, String shardId, String cluster, String service) {
        var flagsConfig = this.flagsConfig;
        if (flagsConfig == null) {
            throw new ConfigNotInitialized();
        }
        return flagsConfig.define(flag, project, shardId, cluster, service, service);
    }

    private Int2ObjectMap<FeatureFlags> ensureConfigLoaded() {
        var snapshot = flagsByNumId;
        if (snapshot == null) {
            throw new ConfigNotInitialized();
        }

        return snapshot;
    }

    private FeatureFlags flags(ShardConfDetailed shard) {
        var flagsConfig = this.flagsConfig;
        if (flagsConfig == null) {
            throw new ConfigNotInitialized();
        }
        return flagsConfig.getFlags(
                shard.getProjectId(),
                shard.getId(),
                shard.getCluster().getId(),
                shard.getService().getId(),
                shard.getService().getRaw().getServiceProvider());
    }

    private FeatureFlags flags(String projectId, String serviceProvider) {
        var flagsConfig = this.flagsConfig;
        if (flagsConfig == null) {
            throw new ConfigNotInitialized();
        }
        return flagsConfig.getFlags(projectId, serviceProvider);
    }

    private void updateSnapshot() {
        if (!tasks.addTask()) {
            return;
        }

        while (tasks.fetchTask()) {
            try {
                flagsByNumId = prepareSnapshot();
                metrics.update(Statistics.of(flagsByNumId));
            } catch (Throwable e) {
                logger.error("Failed update feature flags snapshot: ", e);
            }
        }
    }

    @Nullable
    private Int2ObjectMap<FeatureFlags> prepareSnapshot() {
        var config = this.config;
        if (config == null) {
            return null;
        }

        var flagsConfig = this.flagsConfig;
        if (flagsConfig == null) {
            return null;
        }

        var defaultFlags = flagsConfig.getDefaultFlags();
        var flags = new FeatureFlags();
        var it = config.getCorrectShardsStream().iterator();
        var flagsByNumId = new Int2ObjectOpenHashMap<FeatureFlags>();
        while (it.hasNext()) {
            var shard = it.next();
            var serviceProviderId = shard.getService().getRaw().getServiceProvider();
            var serviceFlags = flagsConfig.getServiceProviderFlags(serviceProviderId);

            if (!flagsConfig.isFlagsDefine(shard.getProjectId())) {
                if (serviceFlags.isEmpty()) {
                    flagsByNumId.put(shard.getNumId(), defaultFlags);
                    continue;
                } else if (defaultFlags.isEmpty()) {
                    flagsByNumId.put(shard.getNumId(), serviceFlags);
                    continue;
                }
            }

            flagsConfig.combineFlags(flags, shard.getProjectId(), shard.getId(), shard.getCluster().getId(), shard.getService().getId(), serviceProviderId);
            if (!flags.isEmpty()) {
                flagsByNumId.put(shard.getNumId(), flags);
                flags = new FeatureFlags();
            } else {
                flagsByNumId.put(shard.getNumId(), FeatureFlags.EMPTY);
                flags.clear();
            }
        }
        return flagsByNumId;
    }

    private <T> void saveStateToFile(String fileName, T config) {
        try {
            storage.save(fileName, config, mapper::writeValueAsString);
        } catch (Throwable e) {
            logger.error("Error while writing state file " + fileName, e);
        }
    }

    @Nullable
    private <T> T loadStateFromFile(String fileName, Class<T> clazz) {
        try {
            return storage.load(fileName, state -> mapper.readValue(state, clazz));
        } catch (Throwable e) {
            logger.error("Error while reading state file " + storage + "/" + fileName, e);
            return null;
        }
    }

    private static class Statistics {
        EnumMapToLong<FeatureFlag> enable = new EnumMapToLong<>(FeatureFlag.class);
        EnumMapToLong<FeatureFlag> disable = new EnumMapToLong<>(FeatureFlag.class);
        EnumMapToLong<FeatureFlag> undefined = new EnumMapToLong<>(FeatureFlag.class);

        public static Statistics of(@Nullable Int2ObjectMap<FeatureFlags> flagsByNumId) {
            if (flagsByNumId == null) {
                return new Statistics();
            }

            var result = new Statistics();
            for (var entry : flagsByNumId.int2ObjectEntrySet()) {
                var flags = entry.getValue();
                for (FeatureFlag flag : FeatureFlag.values()) {
                    if (!flags.isDefined(flag)) {
                        result.undefined.incrementAndGet(flag);
                    } else if (flags.hasFlag(flag)) {
                        result.enable.incrementAndGet(flag);
                    } else {
                        result.disable.incrementAndGet(flag);
                    }
                }
            }
            return result;
        }
    }

    private static class Metrics {
        private EnumMap<FeatureFlag, GaugeInt64> enable;
        private EnumMap<FeatureFlag, GaugeInt64> disable;
        private EnumMap<FeatureFlag, GaugeInt64> undefined;

        public Metrics(MetricRegistry registry) {
            enable = enumsMap(registry, "enable");
            disable = enumsMap(registry, "disable");
            undefined = enumsMap(registry, "undefined");
        }

        private void update(Statistics stats) {
            for (FeatureFlag flag : FeatureFlag.values()) {
                enable.get(flag).set(stats.enable.get(flag));
                disable.get(flag).set(stats.disable.get(flag));
                undefined.get(flag).set(stats.undefined.get(flag));
            }
        }

        private static EnumMap<FeatureFlag, GaugeInt64> enumsMap(MetricRegistry registry, String name) {
            return EnumMetrics.gaugesInt64(FeatureFlag.class, registry, "featureFlags."+name, "flag");
        }
    }
}
