package ru.yandex.solomon.core.conf.flags;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Stream;

import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
abstract class FeatureFlagsOperation {
    private final SolomonConfWithContext config;

    public FeatureFlagsOperation(SolomonConfWithContext config) {
        this.config = config;
    }

    CompletableFuture<?> apply(String projectId, String clusterId, String serviceId, String shardId, String serviceProvider) {
        if (Stream.of(projectId, clusterId, serviceId, shardId, serviceProvider).allMatch(String::isEmpty)) {
            return changeProject(projectId);
        }

        List<Selector> selectors = new ArrayList<>();
        if (!projectId.isEmpty()) {
            var selector = selector("projectId", projectId);
            if (selector.isExact() && Stream.of(clusterId, serviceId, shardId).allMatch(String::isEmpty)) {
                return changeProject(projectId);
            }
            selectors.add(selector);
        }

        if (!serviceProvider.isEmpty()) {
            var selector = selector("serviceProvider", serviceProvider);
            if (selector.isExact() && Stream.of(projectId, clusterId, serviceId, shardId).allMatch(String::isEmpty)) {
                return changeServiceProvider(serviceProvider);
            }
            selectors.add(selector);
        }

        if (!clusterId.isEmpty()) {
            selectors.add(selector("clusterId", clusterId));
        }

        if (!serviceId.isEmpty()) {
            selectors.add(selector("serviceId", serviceId));
        }

        if (!shardId.isEmpty()) {
            selectors.add(selector("shardId", shardId));
        }

        if (selectors.isEmpty()) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Not specified selectors"));
        }

        if (!projectId.isEmpty() && selectors.size() == 2 && isExactMatch(selectors)) {
            if (!clusterId.isEmpty()) {
                return changeCluster(projectId, clusterId);
            }

            if (!serviceId.isEmpty()) {
                return changeService(projectId, serviceId);
            }

            if (!shardId.isEmpty()) {
                return changeShard(projectId, shardId);
            }
        }

        var it = config.getCorrectShardsStream().iterator();
        AsyncActorBody body = () -> {
            while (it.hasNext()) {
                var shard = it.next();
                if (isMatch(selectors, shard)) {
                    return changeShard(shard.getProjectId(), shard.getId());
                }
            }

            return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 1);
        return runner.start();
    }

    protected abstract CompletableFuture<?> changeProject(String projectId);
    protected abstract CompletableFuture<?> changeServiceProvider(String serviceProvider);
    protected abstract CompletableFuture<?> changeCluster(String projectId, String clusterId);
    protected abstract CompletableFuture<?> changeService(String projectId, String serviceId);
    protected abstract CompletableFuture<?> changeShard(String projectId, String shardId);

    private boolean isMatch(List<Selector> selectors, ShardConfDetailed shard) {
        for (var selector : selectors) {
            if (!isMatch(selector, shard)) {
                return false;
            }
        }
        return true;
    }

    private boolean isMatch(Selector selector, ShardConfDetailed shard) {
        switch (selector.getKey()) {
            case "projectId":
                return selector.match(shard.getProjectId());
            case "clusterId":
                return selector.match(shard.getCluster().getId());
            case "serviceId":
                return selector.match(shard.getService().getId());
            case "shardId":
                return selector.match(shard.getId());
            case "serviceProvider":
                return selector.match(shard.getService().getRaw().getServiceProvider());
            default:
                return false;
        }
    }

    private boolean isExactMatch(List<Selector> selectors) {
        for (var selector : selectors) {
            if (!selector.isExact()) {
                return false;
            }
        }
        return true;
    }

    private static Selector selector(String key, String value) {
        if (value.startsWith("!")) {
            return ru.yandex.solomon.labels.query.Selector.notGlob(key, value.substring(1));
        }
        return ru.yandex.solomon.labels.query.Selector.glob(key, value);
    }
}
