package ru.yandex.solomon.core.conf.flags;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlags;
import ru.yandex.solomon.flags.FeatureFlagsConfig;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.staffOnly.UrlUtils;
import ru.yandex.solomon.staffOnly.html.CssLine;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.WritableToHtml;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class FeatureFlagsPage extends ManagerPageTemplate {
    private static final int RECORDS_LIMIT = 1_000;
    private static final String FILTER_BY_NUM_ID_PARAM = "filterByNumId";
    private static final String FILTER_BY_PROJECT_PARAM = "filterByProject";
    private static final String FILTER_BY_SHARD_PARAM = "filterByShard";
    private static final String FILTER_BY_CLUSTER_PARAM = "filterByCluster";
    private static final String FILTER_BY_SERVICE_PARAM = "filterByService";
    private static final String FILTER_BY_SERVICE_PROVIDER_PARAM = "filterByServiceProvider";
    private static final String FILTER_BY_FLAG_PARAM = "filterByFlag";
    private static final String FILTER_BY_FLAG_STATE_PARAM = "filterByFlagState";
    private static final String SORT_BY_COLUMN_PARAM = "sortBy";
    private final String path;
    private final Map<String, String> params;

    // config snapshot
    private final SolomonConfWithContext config;
    private final FeatureFlagsConfig configFlags;
    private final Int2ObjectMap<FeatureFlags> flagsByNumId;

    // Filters
    private int filterByNumId;
    private String filterByProject;
    private String filterByShardId;
    private String filterByClusterId;
    private String filterByServiceId;
    private String filterByServiceProvider;
    @Nullable
    private FeatureFlag filterByFlag;
    @Nullable
    private FlagState filterByFlagState;

    // Sorts
    private String sortByColumn;
    private boolean sortAsAsc;

    private List<Column<Record>> columns;
    private List<Record> records;

    FeatureFlagsPage(String path, Map<String, String> params, FeatureFlagsHolderImpl holder) {
        super("FeatureFlags");
        this.path = path;
        this.params = params;
        this.config = holder.config;
        this.configFlags = Objects.requireNonNull(holder.flagsConfig);
        this.flagsByNumId = Objects.requireNonNull(holder.flagsByNumId);

        parseParameters();
        prepareColumn();
        prepareRecords();
    }

    private void parseParameters() {
        {
            String str = getParam(FILTER_BY_NUM_ID_PARAM);
            this.filterByNumId = str.isEmpty() ? 0 : Integer.parseUnsignedInt(str);
        }
        this.filterByProject = getParam(FILTER_BY_PROJECT_PARAM);
        this.filterByShardId = getParam(FILTER_BY_SHARD_PARAM);
        this.filterByClusterId = getParam(FILTER_BY_CLUSTER_PARAM);
        this.filterByServiceId = getParam(FILTER_BY_SERVICE_PARAM);
        this.filterByServiceProvider = getParam(FILTER_BY_SERVICE_PROVIDER_PARAM);
        {
            String str = getParam(FILTER_BY_FLAG_PARAM);
            this.filterByFlag = FeatureFlag.byName(str);
        }
        {
            String str = getParam(FILTER_BY_FLAG_STATE_PARAM);
            this.filterByFlagState = str.isEmpty() ? null : FlagState.valueOf(str.toUpperCase());
        }
        {
            String sortByStr = getParam(SORT_BY_COLUMN_PARAM);
            this.sortAsAsc = !sortByStr.startsWith("-");
            this.sortByColumn = sortByStr.startsWith("-") ? sortByStr.substring(1) : sortByStr;
        }
    }

    private String getParam(String name) {
        return params.getOrDefault(name, "").trim();
    }

    private Predicate<Record> prepareFilterPredicate() {
        Predicate<Record> filter = ignore -> true;
        if (filterByNumId != 0) {
            filter = filter.and(record -> record.numId == filterByNumId);
        }

        if (!filterByProject.isEmpty()) {
            var selector = selector("projectId", filterByProject);
            filter = filter.and(record -> selector.match(record.projectId));
        }

        if (!filterByShardId.isEmpty()) {
            var selector = selector("shardId", filterByShardId);
            filter = filter.and(record -> selector.match(record.shardId));
        }

        if (!filterByClusterId.isEmpty()) {
            var selector = selector("clusterId", filterByClusterId);
            filter = filter.and(record -> selector.match(record.clusterId));
        }

        if (!filterByServiceId.isEmpty()) {
            var selector = selector("serviceId", filterByServiceId);
            filter = filter.and(record -> selector.match(record.serviceId));
        }

        if (!filterByServiceProvider.isEmpty()) {
            var selector = selector("serviceProvider", filterByServiceProvider);
            filter = filter.and(record -> selector.match(record.serviceProvider));
        }

        if (filterByFlag != null) {
            if (filterByFlagState != null) {
                filter = filter.and(record -> {
                    var flags = record.flags;
                    switch (filterByFlagState) {
                        case ENABLE:
                            return flags.hasFlag(filterByFlag);
                        case DISABLE:
                            return flags.isDefined(filterByFlag) && !flags.hasFlag(filterByFlag);
                        case DEFINED:
                            return flags.isDefined(filterByFlag);
                        default:
                            return !flags.isDefined(filterByFlag);
                    }
                });
            } else {
                filter = filter.and(record -> record.flags.isDefined(filterByFlag));
            }
        } else {
            if (filterByFlagState != null) {
                filter = filter.and(record -> {
                    var flags = record.flags;
                    switch (filterByFlagState) {
                        case DEFINED:
                            return !flags.isEmpty();
                        case UNDEFINED:
                            return flags.isEmpty();
                        case ENABLE:
                            return flags.hasAnyFlag(true);
                        default:
                            return flags.hasAnyFlag(false);
                    }
                });
            }
        }

        return filter;
    }

    private Selector selector(String key, String value) {
        if (value.startsWith("!")) {
            return Selector.notGlob(key, value.substring(1));
        }
        return Selector.glob(key, value);
    }

    @Nullable
    private Comparator<Record> prepareSorting() {
        if (sortByColumn.isEmpty()) {
            return null;
        }

        var sortBy = columns.stream()
                .filter(column -> column.title.equals(sortByColumn))
                .findFirst()
                .orElse(null);

        if (sortBy == null) {
            return null;
        }

        if (sortAsAsc) {
            return sortBy.comparator;
        } else {
            return sortBy.comparator.reversed();
        }
    }

    @Override
    protected void content() {
        tag("div.row", () -> {
            tag("div.col-md-6", () -> {
                tag("div.well", this::writeFiltersForm);
            });
            tag("div.col-md-6", () -> {
                tag("div.well", this::writeEditForm);
            });
        });
        writeTable();
    }

    private void writeFiltersForm() {
        form(() -> {
            tag("div.form-group", () -> {
                h3("Filters");
                table(() -> {
                    textInput(
                            FILTER_BY_NUM_ID_PARAM,
                            "NumId:",
                            filterByNumId == 0 ? " " : Integer.toUnsignedString(filterByNumId),
                            "unsigned");
                    textInput(
                            FILTER_BY_PROJECT_PARAM,
                            "ProjectId:",
                            filterByProject,
                            "glob");
                    textInput(
                            FILTER_BY_SHARD_PARAM,
                            "ShardId:",
                            filterByShardId,
                            "glob");
                    textInput(
                            FILTER_BY_CLUSTER_PARAM,
                            "ClusterId:",
                            filterByClusterId,
                            "glob");
                    textInput(
                            FILTER_BY_SERVICE_PARAM,
                            "ServiceId:",
                            filterByServiceId,
                            "glob");
                    textInput(
                            FILTER_BY_SERVICE_PROVIDER_PARAM,
                            "ServiceProvider:",
                            filterByServiceProvider,
                            "glob");
                    select(
                            FILTER_BY_FLAG_PARAM,
                            "Flag:",
                            Stream.of(FeatureFlag.values()).map(Enum::name).collect(Collectors.toList()),
                            filterByFlag == null ? " " : filterByFlag.name(),
                            "");
                    select(
                            FILTER_BY_FLAG_STATE_PARAM,
                            "FlagState:",
                            Stream.of(FlagState.values()).map(Enum::name).collect(Collectors.toList()),
                            filterByFlagState == null ? " " : filterByFlagState.name(),
                            "");
                });
            });
            tag("div.form-group", () -> {
                buttonSubmitDefault("Apply");
                write(" ");
                tag("a.btn.btn-warning", new Attr("href", path), () -> write("Clear Filters"));
            });
        }, new Attr("action", path), new Attr("class", "form-horizontal"));
    }

    private void writeEditForm() {
        form(() -> {
            tag("div.form-group", () -> {
                h3("Edit Flags");
                textInput("projectId", "ProjectId:", "", "(glob, empty means default for all projects)");
                textInput("clusterId", "ClusterId:", "", "(glob, e.g solomon_production_gateway)");
                textInput("serviceId", "ServiceId:", "", "(glob, e.g *_compute)");
                textInput("serviceProvider", "ServiceProvider:", "", "(glob, e.g compute)");
                textInput("shardId", "ShardId:", "", "(glob, e.g *_stockpile)");
                tr(() -> {
                    tdText("Flag:");
                    td(() -> {
                        List<String> flags = Stream.of(FeatureFlag.values()).map(Enum::name).collect(Collectors.toList());
                        selectFormControl("flag", "flag", flags, "");
                    });
                });
            });
            tag("div.form-group", () -> {
                tag("button.btn.btn-default type=submit formaction=/featureFlags/Enable", () -> write("Enable"));
                write(" ");
                tag("button.btn.btn-default type=submit formaction=/featureFlags/Disable", () -> write("Disable"));
                write(" ");
                tag("button.btn.btn-default type=submit formaction=/featureFlags/Delete", () -> write("Delete"));
            });
        }, new Attr("class", "form-horizontal"));
    }

    private void select(String id, String name, List<String> values, String defaultValue, String help) {
        tr(() -> {
            thText(name);
            td(() -> {
                selectFormControl(id, id, values, defaultValue);
            });
            if (StringUtils.isNotEmpty(help)) {
                td(() -> {
                    tag("span", HtmlWriter.Attr.cssClass("help-block"), () -> write(help));
                });
            } else {
                td(() -> {
                });
            }
        });
    }

    private void textInput(String id, String name, String value, String help) {
        tr(() -> {
            thText(name);
            td(() -> {
                inputTextfieldFormControl(id, id, value);
            });
            if (StringUtils.isNotEmpty(help)) {
                td(() -> {
                    tag("span", HtmlWriter.Attr.cssClass("help-block"), () -> {
                        write(help);
                    });
                });
            } else {
                td(() -> {
                });
            }
        });
    }

    private void writeTable() {
        var writer = new ManagerWriter(new ManagerWriterContext(), this);
        var columns = hideColumns();
        div(() -> write("Snapshot Age: " + DurationUtils.formatDurationMillis(System.currentTimeMillis() - configFlags.getCreatedAt())));
        div(() -> write("Records: " + records.size()));
        if (records.size() > RECORDS_LIMIT) {
            div(() -> write("Table truncated, add one more filter"));
        }
        tag("table", () -> {
            tr(() -> {
                for (var column : columns) {
                    th(() -> {
                        String title = column.title;
                        if (!title.equals(sortByColumn)) {
                            aHref(hrefWithParam(SORT_BY_COLUMN_PARAM, title), title);
                        } else {
                            String link = sortAsAsc ? "-" + title : title;
                            String icon = sortAsAsc
                                    ? "glyphicon-sort-by-attributes"
                                    : "glyphicon-sort-by-attributes-alt";
                            aHref(hrefWithParam(SORT_BY_COLUMN_PARAM, link), () -> {
                                write(column.title);
                                tag("span", HtmlWriter.Attr.cssClass("glyphicon " + icon));
                            });
                        }
                    });
                }
            });
            int recordCount = 0;
            for (var record : records) {
                if (recordCount++ >= RECORDS_LIMIT) {
                    break;
                }
                tr(() -> {
                    for (var column : columns) {
                        writer.writeCellTd(column.content.apply(record));
                    }
                });
            }
        }, Attr.cssClass("table simple-table2 table-hover table-condensed"), Attr.style(new CssLine("word-break", "break-all")));
    }

    private List<Column<Record>> hideColumns() {
        if (filterByFlag != null) {
            return columns.stream()
                    .filter(column -> {
                        var flag = FeatureFlag.byName(column.title);
                        return flag == null || flag == filterByFlag;
                    })
                    .collect(Collectors.toList());
        }

        EnumSet<FeatureFlag> definedFlags = EnumSet.noneOf(FeatureFlag.class);
        for (var flag : FeatureFlag.values()) {
            for (var record : records) {
                if (record.flags.isDefined(flag)) {
                    definedFlags.add(flag);
                    break;
                }
            }
        }

        return columns.stream()
                .filter(column -> {
                    var flag = FeatureFlag.byName(column.title);
                    return flag == null || definedFlags.contains(flag);
                })
                .collect(Collectors.toList());
    }

    private String hrefWithParam(String name, String value) {
        return UrlUtils.hrefWithParam(name, value, path, params);
    }

    private void prepareRecords() {
        Predicate<Record> filter = prepareFilterPredicate();
        var result = new ArrayList<Record>(flagsByNumId.size());
        for (var entry : flagsByNumId.int2ObjectEntrySet()) {
            int numId = entry.getIntKey();
            var flags = entry.getValue();
            var shardConf = config.getShardByNumIdOrNull(numId);
            if (shardConf == null) {
                continue;
            }

            var shard = shardConf.getConfOrThrow();
            var record = Record.of(flags, shard);
            if (filter.test(record)) {
                result.add(record);
            }
        }

        var comparator = prepareSorting();
        if (comparator != null) {
            result.sort(comparator);
        }
        this.records = result;
    }

    private void prepareColumn() {
        List<Column<Record>> result = new ArrayList<>();
        result.add(Column.of(
                "NumId",
                r -> Integer.toUnsignedString(r.numId),
                Comparator.comparingLong(o -> Integer.toUnsignedLong(o.numId))));
        result.add(Column.of(
                "ProjectId",
                r -> r.projectId,
                Comparator.comparing(o -> o.projectId)));
        result.add(Column.of(
                "ShardId",
                r -> r.shardId,
                Comparator.comparing(o -> o.shardId)));
        result.add(Column.of(
                "ClusterId",
                r -> r.clusterId,
                Comparator.comparing(o -> o.clusterId)));
        result.add(Column.of(
                "ServiceId",
                r -> r.serviceId,
                Comparator.comparing(o -> o.serviceId)));

        for (var flag : FeatureFlag.values()) {
            result.add(Column.of(
                    flag.name(),
                    r -> {
                        var state = r.state(flag);
                        return (WritableToHtml) managerWriter -> {
                            var w = managerWriter.getHtmlWriter();
                            var define = r.define(configFlags, flag);
                            switch (state) {
                                case ENABLE:
                                    w.label("success", state.name());
                                    break;
                                case DISABLE:
                                    w.label("danger", state.name());
                                    break;
                                default:
                                    w.label("default", state.name());
                                    break;
                            }

                            if (define != null) {
                                w.tag("span", HtmlWriter.Attr.cssClass("help-block"), () -> w.write("(" + define + ")"));
                            }
                        };
                    },
                    Comparator.comparing(o -> o.state(flag))
            ));
        }

        this.columns = result;
    }

    public enum FlagState {
        ENABLE, DISABLE, DEFINED, UNDEFINED
    }

    public static FlagState getFlagState(FeatureFlag flag, FeatureFlags flags) {
        if (!flags.isDefined(flag)) {
            return FlagState.UNDEFINED;
        }

        if (flags.hasFlag(flag)) {
            return FlagState.ENABLE;
        }

        return FlagState.DISABLE;
    }

    private static class Record implements TableRecord {
        int numId;
        String projectId;
        String clusterId;
        String serviceId;
        String serviceProvider;
        String shardId;
        FeatureFlags flags;

        public static Record of(FeatureFlags flags, ShardConfDetailed shard) {
            var r = new Record();
            r.numId = shard.getNumId();
            r.projectId = shard.getProjectId();
            r.clusterId = shard.getCluster().getId();
            r.serviceId = shard.getService().getId();
            r.serviceProvider = shard.getService().getRaw().getServiceProvider();
            r.shardId = shard.getId();
            r.flags = flags;
            return r;
        }

        public FlagState state(FeatureFlag flag) {
            return getFlagState(flag, flags);
        }

        @Nullable
        public String define(FeatureFlagsConfig config, FeatureFlag flag) {
            return config.define(flag, projectId, shardId, clusterId, serviceId, serviceProvider);
        }
    }
}
