package ru.yandex.solomon.core.conf.watch;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.SolomonConfManager;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.model.AbstractAuditable;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Stepan Koltsov
 */
@Component
@Import({ SolomonConfManager.class })
@LinkedOnRootPage("Configs Watcher")
public class SolomonConfWatcher implements AutoCloseable {

    private static final Logger logger = LoggerFactory.getLogger(SolomonConfWatcher.class);

    private static final Duration DEFAULT_UPDATE_INTERVAL = Duration.ofSeconds(90);
    private static final Duration MAX_UPDATE_INTERVAL = Duration.ofMinutes(10);

    private final SolomonConfManager confManager;
    private final List<SolomonConfListener> listeners;

    private final ActorRunner actor;
    private final WatchActor<Project> projects;
    private final WatchActor<Cluster> clusters;
    private final WatchActor<Service> services;
    private final WatchActor<Shard> shards;
    private final WatchActor<ServiceProvider> serviceProviders;
    private final List<WatchActor<?>> allWatchers;

    Throwable lastError;
    Instant lastErrorTime = Instant.EPOCH;
    Instant lastSuccessTime = Instant.EPOCH;

    @Autowired
    public SolomonConfWatcher(ThreadPoolProvider threads, SolomonConfManager confManager, List<SolomonConfListener> listeners) {
        this.confManager = confManager;
        this.listeners = listeners;

        var executor = threads.getExecutorService("CpuLowPriority", "");
        var timer = threads.getSchedulerExecutorService();

        this.actor = new ActorRunner(this::act, executor);
        this.projects = new WatchActor<>("project", confManager::getProjects, executor, timer, actor);
        this.clusters = new WatchActor<>("cluster", confManager::getClusters, executor, timer, actor);
        this.services = new WatchActor<>("service", confManager::getServices, executor, timer, actor);
        this.shards = new WatchActor<>("shard", confManager::getShards, executor, timer, actor);
        this.serviceProviders = new WatchActor<>("service_provider", confManager::getServiceProviders, executor, timer, actor);
        this.allWatchers = List.of(projects, clusters, services, shards, serviceProviders);
        for (var watcher : allWatchers) {
            watcher.actor.forcePing();
        }
    }

    private void act() {
        try {
            for (var watcher : allWatchers) {
                if (watcher.latest == null) {
                    logger.warn("{} still not loaded", watcher.name);
                    return;
                }
            }

            var conf = SolomonConfWithContext.create(createConf());
            for (SolomonConfListener listener : listeners) {
                try {
                    listener.onConfigurationLoad(conf);
                } catch (Throwable e) {
                    ExceptionUtils.uncaughtException(e);
                }
            }
            lastSuccessTime = Instant.now();
        } catch (Throwable e) {
            lastError = e;
            lastErrorTime = Instant.now();
            logger.error("cannot load configuration", e);
        }
    }

    private SolomonRawConf createConf() {
        var serviceProviders = this.serviceProviders.latest;
        var projects = this.projects.latest;
        var clusters = this.clusters.latest;
        var services = this.services.latest;
        var shards = this.shards.latest;

        var minSnapshotMs = Stream.of(shards, projects, clusters, services, shards)
                .mapToLong(Snapshot::createdAtMs)
                .min()
                .orElse(Long.MAX_VALUE) - TimeUnit.SECONDS.toMillis(5);

        return new SolomonRawConf(
                filterByCreatedAt(serviceProviders.list, minSnapshotMs),
                filterByCreatedAt(projects.list, minSnapshotMs),
                filterByCreatedAt(clusters.list, minSnapshotMs),
                filterByCreatedAt(services.list, minSnapshotMs),
                filterByCreatedAt(shards.list, minSnapshotMs));
    }

    private <T extends AbstractAuditable> List<T> filterByCreatedAt(List<T> list, long tsMillis) {
        return list.stream()
                .filter(e -> e.getCreatedAtMillis() < tsMillis)
                .collect(Collectors.toList());
    }

    @Override
    public void close() throws Exception {
        for (var watcher : allWatchers) {
            watcher.close();
        }
    }

    private static final class WatchActor<T> implements AutoCloseable {
        private final Supplier<CompletableFuture<List<T>>> supplier;
        private final PingActorRunner actor;
        private final ActorRunner rootActor;
        final String name;
        volatile Snapshot<T> latest;

        public WatchActor(
                String name,
                Supplier<CompletableFuture<List<T>>> supplier,
                Executor executor,
                ScheduledExecutorService timer,
                ActorRunner root)
        {
            this.name = name;
            this.supplier = supplier;
            this.actor = PingActorRunner.newBuilder()
                    .operation("watch_config_" + name)
                    .onPing(this::act)
                    .timer(timer)
                    .executor(executor)
                    .backoffDelay(Duration.ofSeconds(1))
                    .pingInterval(DEFAULT_UPDATE_INTERVAL)
                    .backoffMaxDelay(MAX_UPDATE_INTERVAL)
                    .build();
            this.rootActor = root;
        }

        private CompletableFuture<Void> act(int attempt) {
            long createdAtMs = System.currentTimeMillis();
            return supplier.get()
                    .thenAccept(result -> {
                        this.latest = new Snapshot<>(result, createdAtMs);
                        long tookMs = System.currentTimeMillis() - createdAtMs;
                        logger.info("conf reloaded {}={}, took {}", name, result.size(), DurationUtils.formatDurationMillis(tookMs));
                        rootActor.schedule();
                    });
        }

        @Override
        public void close() {
            actor.close();
        }
    }

    private record Snapshot<T>(List<T> list, long createdAtMs){}
}
