package ru.yandex.solomon.core.db.dao;

import java.util.Optional;
import java.util.concurrent.Executor;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.YdbClientsContext;
import ru.yandex.solomon.core.db.dao.client.GrpcProjectManagerLegacyClient;
import ru.yandex.solomon.core.db.dao.client.YdbOrClientProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbAgentDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClusterFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClustersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbDashboardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbFolderMenuDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbGraphsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectMenuDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectSettingsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbQuotasDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServicesDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbUserSettingsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbViewHistoryDao;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;


/**
 * @author snoop
 */
@SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
@Configuration
@Import({
    YdbClientsContext.class,
})
public class ConfigDaoContext {

    private static final String CONFIG_DIR = "Config";
    private static final String VERSION = "V2";

    public static final String USER_SETTINGS_TABLE = "UserSettings";
    public static final String PROJECT_TABLE = "Project";
    public static final String PROJECT_FLAGS_TABLE = PROJECT_TABLE + "Flags";
    public static final String PROJECT_MENU_TABLE = "ProjectMenu";
    public static final String FOLDER_MENU = "FolderMenu";
    public static final String PROJECT_SETTINGS_TABLE = "ProjectSettings";
    public static final String DASHBOARD_TABLE = "Dashboard";
    public static final String GRAPH_TABLE = "Graph";
    public static final String CLUSTER_TABLE = "Cluster";
    public static final String CLUSTER_FLAGS_TABLE = CLUSTER_TABLE + "Flags";
    public static final String SERVICE_TABLE = "Service";
    public static final String SERVICE_FLAGS_TABLE = SERVICE_TABLE + "Flags";
    public static final String SHARD_TABLE = "Shard";
    public static final String SHARD_FLAGS_TABLE = SHARD_TABLE + "Flags";
    public static final String VIEW_HISTORY_TABLE = "ViewHistory";
    public static final String AGENT_TABLE = "Agent";
    public static final String QUOTA_TABLE = "Quota";
    public static final String SERVICE_PROVIDER_TABLE = "ServiceProvider";

    private final TableClient tableClient;
    @Nullable
    private final MetricRegistry metricRegistry;
    private final String ydbPath;
    private final FeatureFlagsHolder featureFlagsHolder;
    private final ObjectMapper objectMapper;
    private final Executor executor;

    @Autowired
    public ConfigDaoContext(
            @Qualifier("globalTableClient") TableClient tableClient,
            @Qualifier("configKikimrRootPath") String rootSchema,
            @Nullable MetricRegistry metricRegistry,
            FeatureFlagsHolder featureFlagsHolder,
            ThreadPoolProvider threads)
    {
        this.tableClient = tableClient;
        this.metricRegistry = metricRegistry;
        this.ydbPath = rootSchema + "/" + CONFIG_DIR + "/" + VERSION;
        this.featureFlagsHolder = featureFlagsHolder;
        this.objectMapper = createObjectMapper();
        this.executor = threads.getExecutorService("CpuLowPriority", "");
    }

    @Bean
    public ProjectsDao projectsDao(Optional<GrpcProjectManagerLegacyClient> projectManagerClient) {
        String tablePath = ydbPath + "/" + PROJECT_TABLE;
        var dao = new YdbProjectsDao(tableClient, tablePath, objectMapper, executor);
        var result = new YdbOrClientProjectsDao(featureFlagsHolder, dao, projectManagerClient);
        return measure(result, ProjectsDao.class);
    }

    @Bean
    public ProjectMenuDao projectMenuDao() {
        String tablePath = ydbPath + "/" + PROJECT_MENU_TABLE;
        var dao = new YdbProjectMenuDao(tableClient, tablePath, objectMapper);
        return measure(dao, ProjectMenuDao.class);
    }

    @Bean
    public FolderMenuDao folderMenuDao() {
        String tablePath = ydbPath + "/" + FOLDER_MENU;
        var dao = new YdbFolderMenuDao(tableClient, tablePath, objectMapper);
        return measure(dao, FolderMenuDao.class);
    }

    @Bean
    public ProjectSettingsDao projectSettingsDao() {
        String tablePath = ydbPath + "/" + PROJECT_SETTINGS_TABLE;
        var dao = new YdbProjectSettingsDao(tableClient, tablePath, objectMapper);
        return measure(dao, ProjectSettingsDao.class);
    }

    @Bean
    public DashboardsDao dashboardsDao() {
        String tablePath = ydbPath + "/" + DASHBOARD_TABLE;
        var dao = new YdbDashboardsDao(tableClient, tablePath, objectMapper);
        return measure(dao, DashboardsDao.class);
    }

    @Bean
    public GraphsDao graphsDao() {
        String tablePath = ydbPath + "/" + GRAPH_TABLE;
        var dao = new YdbGraphsDao(tableClient, tablePath, objectMapper);
        return measure(dao, GraphsDao.class);
    }

    @Bean
    public ClustersDao clustersDao() {
        String tablePath = ydbPath + "/" + CLUSTER_TABLE;
        var dao = new YdbClustersDao(tableClient, tablePath, objectMapper, executor);
        return measure(dao, ClustersDao.class);
    }

    @Bean
    public ServicesDao servicesDao() {
        String tablePath = ydbPath + "/" + SERVICE_TABLE;
        var dao = new YdbServicesDao(tableClient, tablePath, objectMapper, executor);
        return measure(dao, ServicesDao.class);
    }

    @Bean
    public ShardsDao shardsDao() {
        String tablePath = ydbPath + "/" + SHARD_TABLE;
        var dao = new YdbShardsDao(tableClient, tablePath, objectMapper, executor);
        return measure(dao, ShardsDao.class);
    }

    @Bean
    public ViewHistoryDao viewHistoryDao() {
        String tablePath = ydbPath + "/" + VIEW_HISTORY_TABLE;
        var dao = new YdbViewHistoryDao(tableClient, tablePath);
        return measure(dao, ViewHistoryDao.class);
    }

    @Bean
    public UserSettingsDao userSettingsDao() {
        String tablePath = ydbPath + "/" + USER_SETTINGS_TABLE;
        var dao = new YdbUserSettingsDao(tableClient, tablePath, objectMapper);
        return measure(dao, UserSettingsDao.class);
    }

    @Bean
    public AgentDao agentDao() {
        String tablePath = ydbPath + "/" + AGENT_TABLE;
        var dao = new YdbAgentDao(tableClient, tablePath, objectMapper);
        return measure(dao, AgentDao.class);
    }

    @Bean
    public QuotasDao quotasDao() {
        String tablePath = ydbPath + "/" + QUOTA_TABLE;
        var dao = new YdbQuotasDao(tableClient, tablePath);
        return measure(dao, QuotasDao.class);
    }

    @Bean
    public ProjectFlagsDao projectFlagsDao() {
        String tablePath = ydbPath + "/" + PROJECT_FLAGS_TABLE;
        var dao = new YdbProjectFlagsDao(tableClient, tablePath);
        return measure(dao, ProjectFlagsDao.class);
    }

    @Bean
    public ServiceFlagsDao serviceFlagsDao() {
        String tablePath = ydbPath + "/" + SERVICE_FLAGS_TABLE;
        var dao = new YdbServiceFlagsDao(tableClient, tablePath);
        return measure(dao, ServiceFlagsDao.class);
    }

    @Bean
    public ClusterFlagsDao clusterFlagsDao() {
        String tablePath = ydbPath + "/" + CLUSTER_FLAGS_TABLE;
        var dao = new YdbClusterFlagsDao(tableClient, tablePath);
        return measure(dao, ClusterFlagsDao.class);
    }

    @Bean
    public ShardFlagsDao shardFlagsDao() {
        String tablePath = ydbPath + "/" + SHARD_FLAGS_TABLE;
        var dao = new YdbShardFlagsDao(tableClient, tablePath);
        return measure(dao, ShardFlagsDao.class);
    }

    @Bean
    public ServiceProvidersDao serviceProvidersDao() {
        String tablePath = ydbPath + "/" + SERVICE_PROVIDER_TABLE;
        var dao = new YdbServiceProvidersDao(tableClient, tablePath, objectMapper);
        dao.migrateSchema();
        return measure(dao, ServiceProvidersDao.class);
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass) {
        return metricRegistry == null ? dao : DaoMetricsProxy.of(dao, daoClass, metricRegistry);
    }

    public static ObjectMapper createObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.disable(SerializationFeature.INDENT_OUTPUT);
        objectMapper.enable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        objectMapper.enable(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_USING_DEFAULT_VALUE);
        objectMapper.disable(MapperFeature.AUTO_DETECT_CREATORS);
        objectMapper.disable(MapperFeature.AUTO_DETECT_FIELDS);
        objectMapper.disable(MapperFeature.AUTO_DETECT_GETTERS);
        objectMapper.disable(MapperFeature.AUTO_DETECT_IS_GETTERS);
        return objectMapper;
    }
}
