package ru.yandex.solomon.core.db.dao;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.core.db.SchemaAwareDao;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.selfmon.mon.NameAlias;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;


/**
 * @author snoop
 */
public interface DashboardsDao extends SchemaAwareDao {

    /**
     * Inserts the given dashboard into the database.
     *
     * @param dashboard dashboard to insert
     * @return true if insert was successful
     *     false if DB already contains the dashboard with such id.
     */
    CompletableFuture<Boolean> insert(Dashboard dashboard);

    /**
     * Finds a dashboard by unique project and dashboard identifiers.
     *
     * @param projectId   project id
     * @param folderId    folder id
     * @param dashboardId id of looking for dashboard
     * @return non empty optional if the record was found
     *     empty optional otherwise.
     */
    CompletableFuture<Optional<Dashboard>> findOne(String projectId, String folderId, String dashboardId);

    /**
     * Finds limited page of project's dashboards in the database. Offset and page size considered to use default
     * sort order (natural order by {@code id} field).
     *
     * @param projectId project identifier to find dashboards for
     * @param folderId  folder id
     * @param pageOpts  page options
     * @param text      text to filter dashbobards by id and name
     * @return paged result with found dashboards
     */
    CompletableFuture<PagedResult<Dashboard>> findByProjectId(String projectId, String folderId, PageOptions pageOpts, String text);

    /**
     * Finds limited page of all Solomon dashboards in the database.
     *
     * @param pageOpts  page options
     * @param text      text to filter dashbobards by id and name
     * @return paged result with found dashboards
     */
    @NameAlias("findAll.paged")
    CompletableFuture<PagedResult<Dashboard>> findAll(PageOptions pageOpts, String text);

    /**
     * Partially updates a dashboard in the database. To protect from conflicting operations update is
     * performed iff {@code id} and {@code version} number are matched with the record in the database. Please note that
     * some fields are intentionally never get updated. Here they are:
     * <p>
     * - projectId    (dashboard cannot be moved from one project to another)
     * - createdAt    (updating after creation is meaningless)
     * - version      (only increments)
     *
     * @param dashboard dashboard to update
     * @return updated dashboard or
     *     empty optional if there is no dashboard with given id and version number.
     */
    CompletableFuture<Optional<Dashboard>> partialUpdate(Dashboard dashboard);

    CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String dashboardId);

    /**
     * Finds whether dashboard with {@code projectId, @code dashboardId} is exists in the database.
     *
     * @param projectId   project unique identifier
     * @param folderId
     * @param dashboardId dashboard unique identifier
     * @return true if dashboard is exists
     *     false otherwise
     */
    CompletableFuture<Boolean> exists(String projectId, String folderId, String dashboardId);

    CompletableFuture<List<Dashboard>> findAll();

    CompletableFuture<Void> deleteByProjectId(String projectId, String folderId);

    CompletableFuture<List<ShortGraphOrDashboardConf>> findByProjectIdShorted(String projectId, String folderId);
}
