package ru.yandex.solomon.core.db.dao;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.core.db.SchemaAwareDao;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.selfmon.mon.NameAlias;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;


/**
 * @author snoop
 */
public interface GraphsDao extends SchemaAwareDao {

    /**
     * Inserts the given graph into the database.
     *
     * @param graph graph to insert
     * @return true if insert was successful
     *     false if the database already contains a graph with such id.
     */
    CompletableFuture<Boolean> insert(Graph graph);

    /**
     * Finds a graph by the provided identifier.
     *
     * @param projectId project unique id
     * @param folderId  folder id
     * @param graphId   id of looking for graph
     * @return non empty optional if record was found
     *     empty optional otherwise.
     */
    CompletableFuture<Optional<Graph>> findOne(String projectId, String folderId, String graphId);

    /**
     * Finds limited page of project's graphs in the database. Offset and page size considered to use default
     * sort order (natural order by {@code id} field).
     *
     * @param projectId project identifier to find graphs for
     * @param folderId  folder id
     * @param pageOpts  page options
     * @param text      text to filter graphs by id or name
     * @return paged result with found graphs
     */
    CompletableFuture<PagedResult<Graph>> findByProjectId(
            String projectId,
            String folderId,
            PageOptions pageOpts,
            String text);

    /**
     * Finds limited page of all Solomons graphs in the database.
     * It will be removed after full Solomon redesign.
     *
     * @param pageOpts  page options
     * @param text      text to filter graphs by id or name
     * @return paged result with found graphs
     */
    @NameAlias("findAll.paged")
    CompletableFuture<PagedResult<Graph>> findAll(PageOptions pageOpts, String text);

    /**
     * Partially updates graph in the database. To protect from conflicting operations update is
     * performed iff {@code id} and {@code version} number are matched with a record in the database. Please note that
     * some fields are intentionally never get updated. Here they are:
     * <p>
     * - projectId    (graph cannot be moved from one project to another)
     * - createdAt    (updating after creation is meaningless)
     * - version      (only increments)
     *
     * @param graph graph to update
     * @return updated graph or
     *     empty optional if there is no graph with given id and version number.
     */
    CompletableFuture<Optional<Graph>> partialUpdate(Graph graph);

    CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String graphId);

    /**
     * Finds whether graph with {@code id} is exists in the database, with {@code deleted} flag equals to given one.
     *
     * @param projectId project unique identifier
     * @param folderId  folder id
     * @param graphId   graph unique identifier
     * @return true if graph is exists
     *     false otherwise
     */
    CompletableFuture<Boolean> exists(String projectId, String folderId, String graphId);

    CompletableFuture<List<Graph>> findAll();

    CompletableFuture<Void> deleteByProjectId(String projectId, String folderId);

    CompletableFuture<List<ShortGraphOrDashboardConf>> findByProjectIdShorted(String projectId, String folderId);
}
