package ru.yandex.solomon.core.db.dao;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.core.db.SchemaAwareDao;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.selfmon.mon.NameAlias;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;


/**
 * @author snoop
 */
public interface ServicesDao extends SchemaAwareDao {

    /**
     * Inserts the given service into the database.
     *
     * @param service service to insert
     * @return true if insert was successful
     *     false if DB already contains record with such id.
     */
    CompletableFuture<Boolean> insert(Service service);

    /**
     * Finds a service by the provided identifier.
     *
     * @param projectId project unique id
     * @param folderId  folder unique id
     * @param serviceId id of looking for service
     * @return non empty optional if record was found
     *     empty optional otherwise.
     */
    CompletableFuture<Optional<Service>> findOne(String projectId, String folderId, String serviceId);

    /**
     * Finds a limited page of project's services in the database. Offset and page size considered to use default
     * sort order (natural order by {@code id} field).
     *
     * @param projectId project identifier to find services for
     * @param folderId  folder identifier to find services for
     * @param pageOpts  page options
     * @param monitoringModel service monitoring model
     * @return paged result with found services
     */
    CompletableFuture<PagedResult<Service>> findByProjectId(String projectId, String folderId, PageOptions pageOpts, String text, ShardSettings.Type monitoringModel);

    /**
     * Finds a limited page of project's services in the database. Offset and page size considered to use default
     * sort order (natural order by {@code id} field).
     *
     * @param projectId project identifier to find services for
     * @param folderId  folder identifier to find services for
     * @param pageSize  page size
     * @param pageToken  page token/offset
     * @param text  filter services by name
     * @return paged result with found services
     */
    CompletableFuture<TokenBasePage<Service>> findByProjectIdPaged(String projectId, String folderId, int pageSize, String pageToken, String text);

    /**
     * Finds all active services
     *
     * @return list of found services
     */
    CompletableFuture<List<Service>> findAll();

    /**
     * Finds limited page of all Solomon services in the database.
     *
     * @param pageOpts  page options
     * @param text      text to filter services by id and name
     * @return paged result with found services
     */
    @NameAlias("findAll.paged")
    CompletableFuture<PagedResult<Service>> findAll(PageOptions pageOpts, String text);

    /**
     * Partially updates the service in the database. To protect from conflicting operations update is
     * performed iff {@code id} and {@code version} number are matched with a record in the database. Please note that
     * some fields are intentionally never get updated. Here they are:
     * <p>
     * - projectId    (service cannot be moved from one project to another)
     * - createdAt    (updating after creation is meaningless)
     * - version      (only increments)
     *
     * @param service service to update
     * @return updated service or
     *     empty optional if there is no service with given id and version number.
     */
    CompletableFuture<Optional<Service>> partialUpdate(Service service);

    CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String serviceId);

    CompletableFuture<Void> deleteByProjectId(String projectId, String folderId);

    /**
     * Finds whether service with {@code id} is exists in the database.
     *
     * @param projectId project unique identifier
     * @param folderId  folder unique identifier
     * @param serviceId service unique identifier
     * @return true if service is exists
     *     false otherwise
     */
    CompletableFuture<Boolean> exists(String projectId, String folderId, String serviceId);
}
