package ru.yandex.solomon.core.db.dao;

import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;

import ru.yandex.solomon.core.db.SchemaAwareDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.selfmon.mon.NameAlias;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;


/**
 * @author snoop
 */
public interface ShardsDao extends SchemaAwareDao {

    CompletableFuture<Boolean> insert(Shard shard);

    @NameAlias("findOne.project.shard")
    CompletableFuture<Optional<Shard>> findOne(String projectId, String folderId, String shardId);

    @NameAlias("findOne.project.cluster.service")
    CompletableFuture<Optional<Shard>> findByShardKey(String projectId, String clusterName, String serviceName);

    CompletableFuture<PagedResult<Shard>> findByProjectId(String projectId, String folderId, PageOptions pageOpts, EnumSet<ShardState> state, String text);

    @NameAlias("findByProjectId.nonPaged")
    CompletableFuture<List<Shard>> findByProjectId(String projectId, String folderId);

    CompletableFuture<TokenBasePage<Shard>> findByProjectIdV3(String projectId, String folderId, int pageSize, String pageToken, String text);

    CompletableFuture<List<Shard>> findAll();

    @NameAlias("findAll.paged")
    CompletableFuture<PagedResult<Shard>> findAll(PageOptions pageOpts, ShardState state, String text);

    /**
     * Finds all active shards
     *
     * @return list of found shards
     */
    CompletableFuture<List<Shard>> findAllNotInactive();

    CompletableFuture<Optional<Shard>> partialUpdate(Shard shard, boolean canUpdateInternals);

    CompletableFuture<Void> patchWithClusterName(String projectId, String clusterId, String clusterName);

    CompletableFuture<Void> patchWithServiceName(String projectId, String serviceId, String serviceName);

    CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String shardId);

    CompletableFuture<Boolean> releaseNumId(String projectId, String shardId, int numId);

    /**
     * Finds all shards associated with cluster.
     *
     * @param projectId project unique identifier
     * @param folderId folder unique identifier
     * @param clusterId cluster unique identifier
     * @return list of shard references with shardId and serviceId and state.
     */
    CompletableFuture<List<Shard>> findByClusterId(String projectId, String folderId, String clusterId);

    CompletableFuture<TokenBasePage<Shard>> findByClusterIdV3(
            String projectId,
            String folderId,
            String clusterId,
            Set<ShardState> states,
            String filter,
            int pageSize,
            String pageToken);

    /**
     * Finds all shards associated with service.
     *
     * @param projectId project unique identifier
     * @param folderId folder unique identifier
     * @param serviceId service unique identifier
     * @return list of shard references with shardId and clusterId and state.
     */
    CompletableFuture<List<Shard>> findByServiceId(String projectId, String folderId, String serviceId);

    CompletableFuture<TokenBasePage<Shard>> findByServiceIdV3(
            String projectId,
            String folderId,
            String serviceId,
            Set<ShardState> states,
            String filter,
            int pageSize,
            String pageToken);

    /**
     * Checks whether shard with tuple ({@code projectId}, {@code shardId}) exists in the database.
     *
     * @param projectId project unique identifier
     * @param folderId  folder unique identifier
     * @param shardId   shard unique identifier
     * @return true if shard is exists
     *     false otherwise
     */
    CompletableFuture<Boolean> exists(String projectId, String folderId, String shardId);

    /**
     * @return all {id -> shardId} mappings
     */
    CompletableFuture<Int2ObjectMap<String>> findAllIdToShardId();
}
