package ru.yandex.solomon.core.db.dao.client;

import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.Throwables;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.cluster.discovery.ClusterDiscoveryImpl;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.project.manager.v3.CreateProjectRequest;
import ru.yandex.project.manager.v3.DeleteOneRequest;
import ru.yandex.project.manager.v3.ExistsRequest;
import ru.yandex.project.manager.v3.FindByIdRequest;
import ru.yandex.project.manager.v3.FindInProjectsRequest;
import ru.yandex.project.manager.v3.FindRequest;
import ru.yandex.project.manager.v3.FindV3Request;
import ru.yandex.project.manager.v3.GetAllRequest;
import ru.yandex.project.manager.v3.PartialUpdateRequest;
import ru.yandex.project.manager.v3.ProjectLegacyServiceGrpc;
import ru.yandex.project.manager.v3.UpsertProjectsRequest;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;
import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Alexey Trushkin
 */
public class GrpcProjectManagerLegacyClient implements ProjectsDao, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(GrpcProjectManagerLegacyClient.class);
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withExceptionFilter(GrpcProjectManagerLegacyClient::needToRetry)
            .withNumRetries(5)
            .withDelay(1_000)
            .withMaxDelay(60_000);
    private final ClusterDiscovery<GrpcTransport> discovery;

    public GrpcProjectManagerLegacyClient(ClusterDiscoveryImpl<GrpcTransport> discovery) {
        this.discovery = discovery;
    }

    @Override
    public CompletableFuture<Boolean> insert(Project project) {
        var request = CreateProjectRequest.newBuilder()
                .setProject(ProjectDtoConverter.toProto(project))
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getInsertMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute insert request", throwable);
                        handleException(throwable);
                    }
                    return response.getResult();
                }));
    }

    @Override
    public CompletableFuture<Optional<Project>> findById(String id) {
        var request = FindByIdRequest.newBuilder()
                .setId(id)
                .build();
        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getFindByIdMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute findById request", throwable);
                        handleException(throwable);
                    }
                    return ProjectDtoConverter.toModel(response.getProject());
                }));
    }

    @Override
    public CompletableFuture<List<Project>> findAllNames() {
        var request = GetAllRequest.newBuilder()
                .build();
        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getGetAllMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute getAll request", throwable);
                        handleException(throwable);
                    }
                    return ProjectDtoConverter.toModel(response.getProjectsList());
                }));
    }

    @Override
    public CompletableFuture<PagedResult<Project>> find(String text, String abcFilter, String login, @Nullable EnumSet<ProjectPermission> filterByPermissions, PageOptions pageOpts) {
        var request = FindRequest.newBuilder()
                .setText(text)
                .setAbcFilter(abcFilter)
                .setLogin(login)
                .addAllFilterByPermissions(filterByPermissions == null
                        ? List.of()
                        : filterByPermissions.stream()
                        .map(Enum::name)
                        .collect(Collectors.toList()))
                .setSize(pageOpts.getSize())
                .setCurrent(pageOpts.getCurrent())
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getFindMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute find request", throwable);
                        handleException(throwable);
                    }
                    return ProjectDtoConverter.toModel(response);
                }));
    }

    @Override
    public CompletableFuture<PagedResult<Project>> findInProjects(String text, String abcFilter, Set<String> projectIds, PageOptions pageOpts) {
        var request = FindInProjectsRequest.newBuilder()
                .setText(text)
                .setAbcFilter(abcFilter)
                .addAllProjectIds(projectIds)
                .setSize(pageOpts.getSize())
                .setCurrent(pageOpts.getCurrent())
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getFindInProjectsMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute findInProjects request", throwable);
                        handleException(throwable);
                    }
                    return ProjectDtoConverter.toModel(response);
                }));
    }

    @Override
    public CompletableFuture<TokenBasePage<Project>> findV3(String text, int pageSize, String pageToken) {
        var request = FindV3Request.newBuilder()
                .setText(text)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getFindV3Method(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute findV3 request", throwable);
                        handleException(throwable);
                    }
                    return ProjectDtoConverter.toModel(response);
                }));
    }

    @Override
    public CompletableFuture<Optional<Project>> partialUpdate(Project project, boolean canChangeOwner, boolean canChangeInternalOptions, boolean canUpdateOldFields) {
        var request = PartialUpdateRequest.newBuilder()
                .setCanChangeInternalOptions(canChangeInternalOptions)
                .setCanChangeOwner(canChangeOwner)
                .setCanUpdateOldFields(canUpdateOldFields)
                .setProject(ProjectDtoConverter.toProto(project))
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getPartialUpdateMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute partialUpdate request", throwable);
                        handleException(throwable);
                    }
                    return ProjectDtoConverter.toModel(response.getProject());
                }));
    }

    @Override
    public CompletableFuture<Void> upsertProjects(List<Project> projects) {
        var request = UpsertProjectsRequest.newBuilder()
                .addAllProject(ProjectDtoConverter.toProto(projects).getProjectsList())
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getUpsertProjectsMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute upsertProjects request", throwable);
                        handleException(throwable);
                    }
                    return null;
                }));
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String id, boolean skipValidation) {
        var request = DeleteOneRequest.newBuilder()
                .setId(id)
                .setSkipValidation(skipValidation)
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getDeleteOneMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute deleteOne request", throwable);
                        handleException(throwable);
                    }
                    return response.getResult();
                }));
    }

    @Override
    public CompletableFuture<Boolean> exists(String id) {
        var request = ExistsRequest.newBuilder()
                .setId(id)
                .build();

        return run(() -> getTransport().unaryCall(ProjectLegacyServiceGrpc.getExistsMethod(), request)
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("cannot execute exists request", throwable);
                        handleException(throwable);
                    }
                    return response.getResult();
                }));
    }


    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return CompletableFuture.failedFuture(new UnsupportedOperationException());
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return CompletableFuture.failedFuture(new UnsupportedOperationException());
    }

    private GrpcTransport getTransport() {
        for (int i = 0; i < 15; i++) {
            var nodeTransport = discovery.getTransport();
            if (nodeTransport != null && nodeTransport.isConnected()) {
                return nodeTransport;
            }
        }
        throw new IllegalStateException("there are no available Fetcher hosts");
    }

    private <T> CompletableFuture<T> run(Supplier<CompletableFuture<T>> taskProducer) {
        return runWithRetries(() -> safeCall(taskProducer::get), RETRY_CONFIG);
    }

    private void handleException(Throwable throwable) {
        throwable = CompletableFutures.unwrapCompletionException(throwable);
        Throwables.throwIfUnchecked(throwable);
        throw new RuntimeException(throwable);
    }

    @Override
    public void close() throws Exception {
        discovery.close();
    }

    private static boolean needToRetry(Throwable throwable) {
        Status status = Status.fromThrowable(throwable);
        return status.getCode() == Status.Code.UNAVAILABLE;
    }

}
