package ru.yandex.solomon.core.db.dao.client;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;

import ru.yandex.project.manager.v3.FindByIdResponse;
import ru.yandex.project.manager.v3.FindResponse;
import ru.yandex.project.manager.v3.FindV3Response;
import ru.yandex.project.manager.v3.GetAllResponse;
import ru.yandex.project.manager.v3.PartialUpdateResponse;
import ru.yandex.project.manager.v3.UpsertProjectsRequest;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ProjectDtoConverter {
    public static ru.yandex.project.manager.v3.Project toProto(Project project) {
        return ru.yandex.project.manager.v3.Project.newBuilder()
                .setId(project.getId())
                .setName(project.getName())
                .setDescription(project.getDescription())
                .setCreatedAt(Timestamps.fromMillis(project.getCreatedAtMillis()))
                .setModifiedAt(Timestamps.fromMillis(project.getUpdatedAtMillis()))
                .setCreatedBy(project.getCreatedBy())
                .setModifiedBy(project.getUpdatedBy())
                .setAbcService(project.getAbcService())
                .setMetricNameLabel(project.getMetricNameLabel())
                .setMetricNameShardsOnly(project.isOnlyMetricNameShards())
                .setOnlyNewFormatReads(project.isOnlyNewFormatReads())
                .setOnlyNewFormatWrites(project.isOnlyNewFormatWrites())
                .putAllLabels(project.getLabels())
                .setOwner(project.getOwner())
                .setOnlyAuthPush(project.isOnlyAuthPush())
                .setOnlyAuthRead(project.isOnlyAuthRead())
                .addAllCanRead(project.getAcl().getCanRead())
                .addAllCanUpdate(project.getAcl().getCanUpdate())
                .addAllCanDelete(project.getAcl().getCanDelete())
                .addAllCanWrite(project.getAcl().getCanWrite())
                .setVersion(project.getVersion())
                .build();
    }

    public static Optional<Project> toModel(ru.yandex.project.manager.v3.Project project) {
        if (project.equals(ru.yandex.project.manager.v3.Project.getDefaultInstance())) {
            return Optional.empty();
        }
        var acl = Acl.of(new HashSet<>(project.getCanReadList()), new HashSet<>(project.getCanUpdateList()), new HashSet<>(project.getCanDeleteList()), new HashSet<>(project.getCanWriteList()));
        return Optional.of(Project.newBuilder()
                .setId(project.getId())
                .setName(project.getName())
                .setDescription(project.getDescription())
                .setCreatedAtMillis(Timestamps.toMillis(project.getCreatedAt()))
                .setUpdatedAtMillis(Timestamps.toMillis(project.getModifiedAt()))
                .setCreatedBy(project.getCreatedBy())
                .setUpdatedBy(project.getModifiedBy())
                .setAbcService(project.getAbcService())
                .setOwner(project.getOwner())
                .setMetricNameLabel(project.getMetricNameLabel())
                .setOnlyMetricNameShards(project.getMetricNameShardsOnly())
                .setOnlyNewFormatReads(project.getOnlyNewFormatReads())
                .setOnlyNewFormatWrites(project.getOnlyNewFormatWrites())
                .setLabels(project.getLabelsMap())
                .setOnlyAuthPush(project.getOnlyAuthPush())
                .setOnlyAuthRead(project.getOnlyAuthRead())
                .setAcl(acl)
                .setVersion(project.getVersion())
                .build());
    }

    public static GetAllResponse toProto(List<Project> projects) {
        return GetAllResponse.newBuilder()
                .addAllProjects(projects.stream()
                        .map(ProjectDtoConverter::toProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public static FindResponse toProto(PagedResult<Project> result) {
        return FindResponse.newBuilder()
                .addAllProjects(result.getResult().stream()
                        .map(ProjectDtoConverter::toProto)
                        .collect(Collectors.toList()))
                .setCurrentPage(result.getCurrentPage())
                .setPagesCount(result.getPagesCount())
                .setTotalCount(result.getTotalCount())
                .setPageSize(result.getPageSize())
                .build();
    }

    public static FindV3Response toProto(TokenBasePage<Project> projectTokenBasePage) {
        return FindV3Response.newBuilder()
                .addAllProjects(projectTokenBasePage.getItems().stream()
                        .map(ProjectDtoConverter::toProto)
                        .collect(Collectors.toList()))
                .setNextPageToken(projectTokenBasePage.getNextPageToken())
                .build();
    }

    public static FindByIdResponse toProto(Optional<Project> projectOptional) {
        return FindByIdResponse.newBuilder()
                .setProject(projectOptional.isEmpty()
                        ? ru.yandex.project.manager.v3.Project.getDefaultInstance()
                        : ProjectDtoConverter.toProto(projectOptional.get()))
                .build();
    }

    public static PartialUpdateResponse toProtoUpdate(Optional<Project> projectOptional) {
        return PartialUpdateResponse.newBuilder()
                .setProject(projectOptional.isEmpty()
                        ? ru.yandex.project.manager.v3.Project.getDefaultInstance()
                        : ProjectDtoConverter.toProto(projectOptional.get()))
                .build();
    }

    public static List<Project> toModel(UpsertProjectsRequest request) {
        return request.getProjectList().stream()
                .map(ProjectDtoConverter::toModel)
                .map(Optional::get)
                .collect(Collectors.toList());
    }

    public static List<Project> toModel(List<ru.yandex.project.manager.v3.Project> projectsList) {
        return projectsList.stream()
                .map(ProjectDtoConverter::toModel)
                .map(Optional::get)
                .collect(Collectors.toList());
    }

    public static PagedResult<Project> toModel(FindResponse response) {
        return new PagedResult<>(toModel(response.getProjectsList()),
                response.getPagesCount(),
                response.getTotalCount(),
                response.getPageSize(),
                response.getCurrentPage());
    }

    public static TokenBasePage<Project> toModel(FindV3Response response) {
        return new TokenBasePage<>(toModel(response.getProjectsList()),
                response.getNextPageToken());
    }

}
