package ru.yandex.solomon.core.db.dao.kikimr;

import java.io.UncheckedIOException;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.ArrayType;
import com.fasterxml.jackson.databind.type.CollectionType;
import org.apache.commons.lang3.StringUtils;


/**
 * @author snoop
 */
public class KikimrDaoSupport {

    private static final Pattern PUNCTATION_PATTERN = Pattern.compile("\\p{Punct}");

    private static final char TSV_DELIMITER = '\t';
    private static final String TSV_DELIMITER_STR = String.valueOf(TSV_DELIMITER);

    private static final char SPACE = ' ';

    private KikimrDaoSupport() {
    }

    public static String toRegularExpression(String source) {
        return toRegularExpression(source, true);
    }

    public static String toRegularExpression(String source, boolean caseInsensitive) {
        if (source.equals("*")) {
            return ".*";
        }

        StringBuilder sb = new StringBuilder();

        if (caseInsensitive) {
            sb.append("(?i)");
        }

        boolean leadingWildcard = source.startsWith("*");
        boolean trailingWildcard = source.endsWith("*");

        String valueToUse = source.substring(leadingWildcard ? 1 : 0,
            trailingWildcard ? source.length() - 1 : source.length());

        if (PUNCTATION_PATTERN.matcher(valueToUse).find()) {
            valueToUse = Pattern.quote(valueToUse);
        }

        if (leadingWildcard) {
            sb.append(".*");
        }
        sb.append(valueToUse);
        if (trailingWildcard) {
            sb.append(".*");
        }

        return sb.toString();
    }

    public static String toTsvRegularExpression(String source) {
        final String CASE_INSENSITIVE_PREFIX = "(?i)";

        if (StringUtils.isBlank(source)) {
            return ".*";
        }

        String baseRegexp = toRegularExpression(source);

        boolean caseSensitive = false;

        if (baseRegexp.startsWith(CASE_INSENSITIVE_PREFIX)) {
            caseSensitive = true;
            baseRegexp = baseRegexp.substring(CASE_INSENSITIVE_PREFIX.length());
        }

        String caseSensitivePrefix = (caseSensitive ? CASE_INSENSITIVE_PREFIX : "");

        String singleRegexp = "^" + baseRegexp + "$";
        String firstRegexp = "^" + baseRegexp + "\\t.*$";
        String lastRegexp = "^.*\\t" + baseRegexp + "$";
        String middleRegexp = "^.*\\t" + baseRegexp + "\\t.*$";

        return caseSensitivePrefix +
            "((" + singleRegexp
            + ")|(" + firstRegexp
            + ")|(" + lastRegexp
            + ")|(" + middleRegexp
            + "))";
    }

    public static String toJson(ObjectMapper objectMapper, Object value) {
        if (value == null) {
            return null;
        }
        try {
            return objectMapper.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static String toTsv(Iterable<String> values) {
        return toTsv(values, false);
    }

    public static String toTsv(Iterable<String> values, boolean replaceTabToSpace) {
        if (values == null) {
            return null;
        }
        return toTsv(StreamSupport.stream(values.spliterator(), false), replaceTabToSpace);
    }

    public static <T> T fromJson(ObjectMapper objectMapper, String jsonText, Class<T> valueClazz) {
        try {
            return objectMapper.readValue(jsonText, valueClazz);
        } catch (java.io.IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Nullable
    public static <T> T fromJsonOrNull(ObjectMapper objectMapper, String jsonText, Class<T> valueClazz) {
        try {
            if (StringUtils.isEmpty(jsonText)) {
                return null;
            }
            return objectMapper.readValue(jsonText, valueClazz);
        } catch (java.io.IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static <T> List<T> fromJson(ObjectMapper objectMapper, String jsonText, TypeReference<List<T>> typeRef) {
        try {
            return objectMapper.readValue(jsonText, typeRef);
        } catch (java.io.IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Map<String, String> fromJsonMap(ObjectMapper objectMapper, String jsonText) {
        if (jsonText.isEmpty()) {
            return Collections.emptyMap();
        }

        try {
            return objectMapper.readValue(jsonText, new TypeReference<Map<String, String>>(){});
        } catch (java.io.IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static <T> T[] fromJsonArray(ObjectMapper objectMapper, String jsonText, Class<T> elementClazz) {
        if (StringUtils.isEmpty(jsonText)) {
            return (T[]) Array.newInstance(elementClazz, 0);
        }
        ArrayType javaType = objectMapper.getTypeFactory().constructArrayType(elementClazz);
        try {
            return objectMapper.readValue(jsonText, javaType);
        } catch (java.io.IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static <T> List<T> fromJsonList(ObjectMapper objectMapper, String jsonText, Class<T> elementClazz) {
        if (StringUtils.isEmpty(jsonText)) {
            return Collections.emptyList();
        }
        CollectionType javaType = objectMapper.getTypeFactory().constructCollectionType(List.class, elementClazz);
        try {
            return objectMapper.readValue(jsonText, javaType);
        } catch (java.io.IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static List<String> listFromTsv(String text) {
        return fromTsv(text, ArrayList::new);
    }

    public static Set<String> setFromTsv(String text) {
        return fromTsv(text, LinkedHashSet::new);
    }

    private static <C extends Collection<String>> C fromTsv(String text, Supplier<C> collectionFactory) {
        if (text == null) {
            return null;
        }
        if (StringUtils.isEmpty(text)) {
            return collectionFactory.get();
        }
        return streamFromTsv(text).collect(Collectors.toCollection(collectionFactory));
    }

    private static String toTsv(Stream<String> stream, boolean replaceTabToSpace) {
        stream = stream.map(Objects::requireNonNull);
        if (replaceTabToSpace) {
            stream = stream.map(v -> v.replace(TSV_DELIMITER, SPACE));
        }
        return stream.collect(Collectors.joining(TSV_DELIMITER_STR));
    }

    private static Stream<String> streamFromTsv(String text) {
        return Arrays.stream(text.split(TSV_DELIMITER_STR));
    }
}
