package ru.yandex.solomon.core.db.dao.kikimr;

import java.util.Map;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.text.StrSubstitutor;

import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.TokenPageOptions;


/**
 * @author snoop
 */
public class QueryText {

    private static final String PAGE_SIZE_PARAM = "page.size";
    private static final String PAGE_OFFSET_PARAM = "page.offset";
    private static final ImmutableMap<String, Object> NO_PAGING_PARAMETERS = ImmutableMap.of(
        PAGE_SIZE_PARAM, Integer.MAX_VALUE,
        PAGE_OFFSET_PARAM, 0L
    );

    private final ImmutableMap<String, String> queryById;

    public QueryText(Map<String, String> queryById) {
        this.queryById = ImmutableMap.copyOf(queryById);
    }

    public String query(String queryId) {
        return Preconditions.checkNotNull(queryById.get(queryId), "cannot find query: %s", queryId);
    }

    public String query(String queryId, Map<String, Object> dynamicParameters) {
        if (dynamicParameters.isEmpty()) {
            return query(queryId);
        }
        return StrSubstitutor.replace(query(queryId), dynamicParameters);
    }

    public static String prepareQuery(String query, Map<String, Object> dynamicParameters) {
        if (dynamicParameters.isEmpty()) {
            return query;
        }
        return StrSubstitutor.replace(query, dynamicParameters);
    }

    public static Map<String, Object> paging(PageOptions pageOptions) {
        return pageOptions.isLimited() ?
            ImmutableMap.of(
                PAGE_SIZE_PARAM, pageOptions.getSize(),
                PAGE_OFFSET_PARAM, pageOptions.getOffset()
            ) : NO_PAGING_PARAMETERS;
    }

    public static Map<String, Object> paging(TokenPageOptions pageOptions) {
        return ImmutableMap.of(
                PAGE_SIZE_PARAM, pageOptions.getSize() + 1,
                PAGE_OFFSET_PARAM, pageOptions.getOffset()
            );
    }

    public static Map<String, Object> pagingToken(TokenPageOptions pageOptions) {
        return ImmutableMap.of(
                PAGE_SIZE_PARAM, pageOptions.getSize() + 1
        );
    }
}
