package ru.yandex.solomon.core.db.dao.memory;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.db.dao.DashboardsDao;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

/**
 * @author Vladimir Gordiychuk
 */
public class InMemoryDashboardsDao implements DashboardsDao {
    private ConcurrentHashMap<Key, Dashboard> map = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return CompletableFuture.supplyAsync(() -> null);
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return CompletableFuture.supplyAsync(() -> null);
    }

    @Override
    public CompletableFuture<Boolean> insert(Dashboard dashboard) {
        return CompletableFuture.supplyAsync(() -> {
            var key = new Key(dashboard.getProjectId(), dashboard.getId());
            return map.putIfAbsent(key, dashboard) == null;
        });
    }

    @Override
    public CompletableFuture<Optional<Dashboard>> findOne(String projectId, String folderId, String dashboardId) {
        return CompletableFuture.supplyAsync(() -> {
            var key = new Key(projectId, dashboardId);
            return Optional.ofNullable(map.get(key))
                    .filter(dashboard -> folderId.isEmpty() || folderId.equals(dashboard.getFolderId()));
        });
    }

    @Override
    public CompletableFuture<PagedResult<Dashboard>> findByProjectId(String projectId, String folderId, PageOptions pageOpts, String text) {
        return CompletableFuture.supplyAsync(() -> {
            var matched = map.values()
                    .stream()
                    .filter(dashboard -> dashboard.getProjectId().equals(projectId))
                    .filter(dashboard -> folderId.isEmpty() || folderId.equals(dashboard.getFolderId()))
                    .filter(dashboard -> StringUtils.containsIgnoreCase(dashboard.getId(), text)
                            || StringUtils.containsIgnoreCase(dashboard.getName(), text))
                    .collect(Collectors.toList());

            return toPageResult(matched, pageOpts);
        });
    }

    @Override
    public CompletableFuture<PagedResult<Dashboard>> findAll(PageOptions pageOpts, String text) {
        return CompletableFuture.supplyAsync(() -> {
            var matched = map.values()
                    .stream()
                    .filter(dashboard -> StringUtils.containsIgnoreCase(dashboard.getId(), text)
                            || StringUtils.containsIgnoreCase(dashboard.getName(), text))
                    .collect(Collectors.toList());

            return toPageResult(matched, pageOpts);
        });
    }

    @Override
    public CompletableFuture<Optional<Dashboard>> partialUpdate(Dashboard dashboard) {
        return findOne(dashboard.getProjectId(), dashboard.getFolderId(), dashboard.getId())
                .thenApply(optional -> {
                    if (optional.isEmpty()) {
                        return Optional.empty();
                    }

                    var prev = optional.get();
                    if (prev.getVersion() != dashboard.getVersion()) {
                        return Optional.empty();
                    }

                    var patched = prev.toBuilder()
                            .setName(dashboard.getName())
                            .setDescription(dashboard.getDescription())
                            .setHeightMultiplier(dashboard.getHeightMultiplier())
                            .setParameters(dashboard.getParameters())
                            .setRows(dashboard.getRows())
                            .setUpdatedAt(dashboard.getUpdatedAt())
                            .setUpdatedBy(dashboard.getUpdatedBy())
                            .setVersion(dashboard.getVersion() + 1)
                            .build();

                    if (map.replace(new Key(dashboard.getProjectId(), dashboard.getId()), prev, patched)) {
                        return Optional.of(patched);
                    }

                    return Optional.empty();
                });
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String dashboardId) {
        return findOne(projectId, folderId, dashboardId)
                .thenApply(opt -> {
                    if (opt.isEmpty()) {
                        return false;
                    }

                    var dashboard = opt.get();
                    return map.remove(new Key(projectId, dashboardId), dashboard);
                });
    }

    @Override
    public CompletableFuture<Boolean> exists(String projectId, String folderId, String dashboardId) {
        return findOne(projectId, folderId, dashboardId).thenApply(Optional::isPresent);
    }

    @Override
    public CompletableFuture<List<Dashboard>> findAll() {
        return CompletableFuture.supplyAsync(() -> List.copyOf(map.values()));
    }

    @Override
    public CompletableFuture<Void> deleteByProjectId(String projectId, String folderId) {
        return CompletableFuture.runAsync(() -> {
            var it = map.values().iterator();
            while (it.hasNext()) {
                var dashboard = it.next();
                if (!dashboard.getProjectId().equals(projectId)) {
                    continue;
                }

                if (!folderId.isEmpty() && !dashboard.getFolderId().equals(folderId)) {
                    continue;
                }

                it.remove();
            }
        });
    }

    @Override
    public CompletableFuture<List<ShortGraphOrDashboardConf>> findByProjectIdShorted(String projectId, String folderId) {
        return findByProjectId(projectId, folderId, PageOptions.ALL, "")
                .thenApply(result -> result.getResult().stream()
                        .map(dashboard -> new ShortGraphOrDashboardConf(dashboard.getId(), projectId, dashboard.getName(), dashboard.getParameters()))
                        .collect(Collectors.toList()));
    }

    private PagedResult<Dashboard> toPageResult(List<Dashboard> matched, PageOptions pageOpts) {
        if (!pageOpts.isLimited()) {
            return PagedResult.of(matched, pageOpts, matched.size());
        }

        var list = matched.subList(pageOpts.getOffset(), Math.min(pageOpts.getOffset() + pageOpts.getSize(), matched.size()));
        return PagedResult.of(list, pageOpts, matched.size());
    }

    private static record Key(String projectId, String id) {
    }
}
