package ru.yandex.solomon.core.db.dao.memory;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.core.db.dao.QuotasDao;
import ru.yandex.solomon.core.db.model.Quota;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class InMemoryQuotasDao implements QuotasDao {

    private static class Key {
        final String namespace;
        final String scopeType;
        @Nullable
        final String scopeId;
        final String indicator;

        public Key(String namespace, String scopeType, @Nullable String scopeId, String indicator) {
            this.namespace = namespace;
            this.scopeType = scopeType;
            this.scopeId = scopeId;
            this.indicator = indicator;
        }

        boolean scopeEquals(String namespace, String scopeType, @Nullable String scopeId) {
            if (!this.namespace.equals(namespace)) {
                return false;
            }
            if (!this.scopeType.equals(scopeType)) {
                return false;
            }
            return this.scopeId == null || Objects.equals(this.scopeId, scopeId);
        }

        boolean indicatorEquals(String namespace, String scopeType, String indicator) {
            if (!this.namespace.equals(namespace)) {
                return false;
            }
            if (!this.scopeType.equals(scopeType)) {
                return false;
            }
            return this.indicator.equals(indicator);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return namespace.equals(key.namespace) &&
                scopeType.equals(key.scopeType) &&
                Objects.equals(scopeId, key.scopeId) &&
                indicator.equals(key.indicator);
        }

        @Override
        public int hashCode() {
            return Objects.hash(namespace, scopeType, scopeId, indicator);
        }

        @Override
        public String toString() {
            return "Key{" +
                "namespace='" + namespace + '\'' +
                ", scopeType='" + scopeType + '\'' +
                ", scopeId='" + scopeId + '\'' +
                ", indicator='" + indicator + '\'' +
                '}';
        }
    }

    private static class Value {
        final Long limit;
        final Instant updatedAt;
        final String updatedBy;

        public Value(Long limit, Instant updatedAt, String updatedBy) {
            this.limit = limit;
            this.updatedAt = updatedAt;
            this.updatedBy = updatedBy;
        }

        @Override
        public String toString() {
            return "Value{" +
                "limit=" + limit +
                ", updatedAt=" + updatedAt +
                ", updatedBy='" + updatedBy + '\'' +
                '}';
        }
    }

    private HashMap<Key, Value> table;

    private static Quota fromRow(Map.Entry<Key, Value> row) {
        return Quota.newBuilder()
            .setNamespace(row.getKey().namespace)
            .setScopeType(row.getKey().scopeType)
            .setScopeId(row.getKey().scopeId)
            .setIndicator(row.getKey().indicator)
            .setUpdatedAt(row.getValue().updatedAt)
            .setUpdatedBy(row.getValue().updatedBy)
            .setLimit(row.getValue().limit)
            .build();
    }

    @Override
    public CompletableFuture<List<Quota>> findAllByNamespace(String namespace) {
        try {
            var result = table.entrySet().stream()
                .filter(row -> row.getKey().namespace.equals(namespace))
                .map(InMemoryQuotasDao::fromRow)
                .collect(Collectors.toList());

            return completedFuture(result);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Quota>> findAllIndicators(String namespace, String scopeType, @Nullable String scopeId) {
        try {
            var result = table.entrySet().stream()
                .filter(row -> row.getKey().scopeEquals(namespace, scopeType, scopeId))
                .map(InMemoryQuotasDao::fromRow)
                .collect(Collectors.toList());

            return completedFuture(result);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> upsert(
            String namespace,
            String scopeType,
            @Nullable String scopeId,
            String indicator,
            long newLimit,
            String updatedBy,
            Instant updatedAt)
    {
        try {
            Key key = new Key(namespace, scopeType, scopeId, indicator);
            Value value = new Value(newLimit, updatedAt, updatedBy);
            table.put(key, value);

            return completedFuture(null);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteOne(String namespace, String scopeType, @Nullable String scopeId, String indicator) {
        try {
            Key key = new Key(namespace, scopeType, scopeId, indicator);
            table.remove(key);

            return completedFuture(null);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> delete(String namespace, String scopeType, String indicator) {
        try {
            table.entrySet().removeIf(row -> row.getKey().indicatorEquals(namespace, scopeType, indicator));

            return completedFuture(null);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        try {
            table = new HashMap<>();
            return completedFuture(null);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        try {
            table = null;
            return completedFuture(null);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }
}
