package ru.yandex.solomon.core.db.dao.memory;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Vladimir Gordiychuk
 */
public class InMemoryShardDao implements ShardsDao {
    public volatile Supplier<CompletableFuture<?>> beforeSupplier;
    private final Map<ShardPk, Shard> shardByKey = new ConcurrentHashMap<>();
    private final Map<Pks, String> shardIdByPks = new ConcurrentHashMap<>();
    private final Map<Integer, String> shardIdByNumId = new ConcurrentHashMap<>();


    public Shard getById(String projectId, String id) {
        return shardByKey.get(new ShardPk(projectId, id));
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return async(() -> {});
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return async(() -> {
            shardByKey.clear();
            shardIdByPks.clear();
            shardIdByNumId.clear();
        });
    }

    @Override
    public CompletableFuture<Boolean> insert(Shard shard) {
        return async(() -> {
            var key = new ShardPk(shard.getProjectId(), shard.getId());
            var pks = new Pks(shard.getProjectId(), shard.getClusterName(), shard.getServiceName());
            var numId = shard.getNumId();

            if (shardByKey.containsKey(key) || shardIdByPks.containsKey(pks) || shardIdByNumId.containsKey(numId)) {
                return Boolean.FALSE;
            }

            shardIdByNumId.put(numId, shard.getId());
            shardIdByPks.put(pks, shard.getId());
            shardByKey.put(key, shard);
            return Boolean.TRUE;
        });
    }

    @Override
    public CompletableFuture<Optional<Shard>> findOne(String projectId, String folderId, String shardId) {
        return async(() -> {
            var key = new ShardPk(projectId, shardId);
            var shard = shardByKey.get(key);
            if (shard == null || (!folderId.isEmpty() && !shard.getFolderId().equals(folderId))) {
                return Optional.empty();
            }

            return Optional.of(shard);
        });
    }

    @Override
    public CompletableFuture<Optional<Shard>> findByShardKey(String projectId, String clusterName, String serviceName) {
        return async(() -> {
            var pks = new Pks(projectId, clusterName, serviceName);
            var id = shardIdByPks.get(pks);
            if (id == null) {
                return Optional.empty();
            }

            var key = new ShardPk(projectId, id);
            return Optional.ofNullable(shardByKey.get(key));
        });
    }

    @Override
    public CompletableFuture<PagedResult<Shard>> findByProjectId(String projectId, String folderId, PageOptions pageOpts, EnumSet<ShardState> state, String text) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<List<Shard>> findByProjectId(String projectId, String folderId) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<TokenBasePage<Shard>> findByProjectIdV3(String projectId, String folderId, int pageSize, String pageToken, String text) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<List<Shard>> findAll() {
        return async(() -> List.copyOf(shardByKey.values()));
    }

    @Override
    public CompletableFuture<PagedResult<Shard>> findAll(PageOptions pageOpts, ShardState state, String text) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<List<Shard>> findAllNotInactive() {
        return notImplemented();
    }

    @Override
    public CompletableFuture<Optional<Shard>> partialUpdate(Shard shard, boolean canUpdateInternals) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<Void> patchWithClusterName(String projectId, String clusterId, String clusterName) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<Void> patchWithServiceName(String projectId, String serviceId, String serviceName) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String shardId) {
        return async(() -> {
            var key = new ShardPk(projectId, shardId);
            var shard = shardByKey.get(key);
            if (shard == null || (!folderId.isEmpty() && !shard.getFolderId().equals(folderId))) {
                return Boolean.FALSE;
            }

            var pks = new Pks(shard.getProjectId(), shard.getClusterName(), shard.getServiceName());
            shardIdByPks.remove(pks, shardId);
            shardByKey.remove(key, shard);
            return Boolean.TRUE;
        });
    }

    @Override
    public CompletableFuture<Boolean> releaseNumId(String projectId, String shardId, int numId) {
        return async(() -> {
            var key = new ShardPk(projectId, shardId);
            var shard = shardByKey.get(key);
            if (shard != null) {
                return Boolean.FALSE;
            }

            if (!shardIdByNumId.containsKey(numId)) {
                return Boolean.TRUE;
            }

            return shardIdByNumId.remove(numId, shardId);
        });
    }

    @Override
    public CompletableFuture<List<Shard>> findByClusterId(String projectId, String folderId, String clusterId) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<TokenBasePage<Shard>> findByClusterIdV3(String projectId, String folderId, String clusterId, Set<ShardState> states, String filter, int pageSize, String pageToken) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<List<Shard>> findByServiceId(String projectId, String folderId, String serviceId) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<TokenBasePage<Shard>> findByServiceIdV3(String projectId, String folderId, String serviceId, Set<ShardState> states, String filter, int pageSize, String pageToken) {
        return notImplemented();
    }

    @Override
    public CompletableFuture<Boolean> exists(String projectId, String folderId, String shardId) {
        return findOne(projectId, folderId, shardId)
                .thenApply(Optional::isPresent);
    }

    @Override
    public CompletableFuture<Int2ObjectMap<String>> findAllIdToShardId() {
        return async(() -> {
            Int2ObjectMap<String> result = new Int2ObjectOpenHashMap<>(shardIdByNumId.size());
            result.putAll(shardIdByNumId);
            return result;
        });
    }

    private <T> CompletableFuture<T> notImplemented() {
        return CompletableFuture.failedFuture(new UnsupportedOperationException("not implemented"));
    }

    private CompletableFuture<Void> async(Runnable fn) {
        return before().thenRunAsync(fn);
    }

    private <T> CompletableFuture<T> async(Supplier<T> fn) {
        return before().thenApplyAsync(ignore -> fn.get());
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return CompletableFuture.completedFuture(null);
        }

        return copy.get();
    }

    private record ShardPk(String projectId, String id) {}
    private record Pks(String projectId, String clusterName, String serviceName){}
}
