package ru.yandex.solomon.core.db.dao.ydb;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.db.dao.AgentDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.Agent;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;

/**
 * @author Max Sherbakov
 */
public class YdbAgentDao implements AgentDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate("agent", Arrays.asList(
        "insert",
        "delete",
        "delete_for_provider",
        "delete_obsolete",
        "find_for_provider"
    ));

    public YdbAgentDao(TableClient tableClient, String path, ObjectMapper objectMapper) {
        this.table = new AgentTable(tableClient, path, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("agent.table.path", path));
    }

    private static final class AgentTable extends YdbTable<String, Agent> {
        private final ObjectMapper objectMapper;

        AgentTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected String getId(Agent agent) { return agent.getHostname(); }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("hostname", PrimitiveType.utf8())
                    .addNullableColumn("dataPort", PrimitiveType.int32())
                    .addNullableColumn("managementPort", PrimitiveType.int32())
                    .addNullableColumn("provider", PrimitiveType.utf8())
                    .addNullableColumn("clusterId", PrimitiveType.utf8())
                    .addNullableColumn("version", PrimitiveType.utf8())
                    .addNullableColumn("lastSeen", PrimitiveType.int64())
                    .addNullableColumn("description", PrimitiveType.utf8())
                    .addNullableColumn("pullIntervalSeconds", PrimitiveType.int32())
                    .setPrimaryKeys("provider", "hostname")
                    .build();
        }

        @Override
        protected Agent mapFull(ResultSetReader r) {
            return partialAgent(Agent.newBuilder(), r)
                .build();
        }

        @Override
        protected Params toParams(Agent agent) {
            return Params.create()
                .put("$hostname", utf8(agent.getHostname()))
                .put("$provider", utf8(agent.getProvider()))
                .put("$dataPort", int32(agent.getDataPort()))
                .put("$managementPort", int32(agent.getManagementPort()))
                .put("$lastSeen", int64(agent.getLastSeen().toEpochMilli()))
                .put("$description", utf8(agent.getDescription()))
                .put("$version", utf8(agent.getVersion()))
                .put("$pullIntervalSeconds", int32(agent.getIntervalSeconds()));
        }

        @Override
        protected Agent mapPartial(ResultSetReader resultSet) {
            return partialAgent(Agent.newBuilder(), resultSet).build();
        }

        private static Agent.Builder partialAgent(Agent.Builder builder, ResultSetReader resultSet) {
            return builder
                .setHostname(resultSet.getColumn("hostname").getUtf8())
                .setProvider(resultSet.getColumn("provider").getUtf8())
                .setDataPort(resultSet.getColumn("dataPort").getInt32())
                .setManagementPort(resultSet.getColumn("managementPort").getInt32())
                .setVersion(resultSet.getColumn("version").getUtf8())
                .setDescription(resultSet.getColumn("description").getUtf8())
                .setPullIntervalSeconds(resultSet.getColumn("pullIntervalSeconds").getInt32())
                .setLastSeen(Instant.ofEpochMilli(resultSet.getColumn("lastSeen").getInt64()));
        }
    }

    private final AgentTable table;
    private final QueryText queryText;

    @Override
    public CompletableFuture<Void> insertOrUpdate(Agent agent) {
        try {
            String query = queryText.query("insert");
            return table.upsertOne(query, agent);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Agent>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<PagedResult<Agent>> findByProvider(String provider, PageOptions pageOpts) {
        try {
            Params params = Params.of("$provider", utf8(provider));
            return table.queryPage(params, pageOpts, opts -> queryText.query("find_for_provider", paging(opts)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteOne(String provider, String url) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$hostname", utf8(url), "$provider", utf8(provider));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteByProvider(String provider) {
        try {
            String query = queryText.query("delete_for_provider");
            Params params = Params.of("$provider", utf8(provider));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Integer> deleteObsolete(Instant deadline) {
        try {
            String query = queryText.query("delete_obsolete");
            Params params = Params.of("$deadline", int64(deadline.getEpochSecond()));
            return table.queryCount(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }
}
