package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.ClusterCloudDnsConf;
import ru.yandex.solomon.core.db.model.ClusterConductorGroupConf;
import ru.yandex.solomon.core.db.model.ClusterConductorTagConf;
import ru.yandex.solomon.core.db.model.ClusterHostListConf;
import ru.yandex.solomon.core.db.model.ClusterHostUrlConf;
import ru.yandex.solomon.core.db.model.ClusterInstanceGroupConf;
import ru.yandex.solomon.core.db.model.ClusterNannyGroupConf;
import ru.yandex.solomon.core.db.model.ClusterNetworkConf;
import ru.yandex.solomon.core.db.model.ClusterQloudGroupConf;
import ru.yandex.solomon.core.db.model.ClusterYpConf;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonList;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonMap;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonOrNull;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;


/**r
 * @author Sergey Polovko
 */
public class YdbClustersDao implements ClustersDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("cluster", Arrays.asList(
        "insert",
        "find",
        "find_for_project",
        "find_for_project_paged",
        "paged_find_all",
        "paged_find_all_lite",
        "update_partial",
        "delete",
        "delete_for_project",
        "exists"
    ));

    private final ClustersTable table;
    private final QueryText queryText;

    public YdbClustersDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper, Executor executor) {
        this.table = new ClustersTable(tableClient, tablePath, objectMapper, executor);
        this.queryText = TEMPLATE.build(Collections.singletonMap("cluster.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Boolean> insert(Cluster cluster) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, cluster);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Cluster>> findOne(String projectId, String folderId, String clusterId) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$id", utf8(clusterId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<PagedResult<Cluster>> findByProjectId(
            String projectId, String folderId, PageOptions pageOpts, String text)
    {
        try {
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId), "$regexp", utf8(toRegularExpression(text)));
            return table.queryPage(params, pageOpts, opts -> {
                return queryText.query("find_for_project", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Cluster>> findByProjectIdPaged(
            String projectId,
            String folderId,
            int pageSize,
            String pageToken,
            String text) {
        try {
            TokenPageOptions pageOpts = new TokenPageOptions(pageSize, pageToken);
            Params params = Params.of(
                    "$projectId", utf8(projectId),
                    "$folderId", utf8(folderId),
                    "$regexp", utf8(toRegularExpression(text))
            );
            return table.queryPage(params, pageOpts, opts -> queryText.query("find_for_project_paged", paging(opts)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Cluster>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<PagedResult<Cluster>> findAll(PageOptions pageOpts, String text) {
        boolean useRegex = StringUtils.isNoneBlank(text);
        if (!pageOpts.isLimited() && !useRegex) {
            return table.queryAll().thenApply(result -> PagedResult.of(result, pageOpts, result.size()));
        }

        try {
            Params params = Params.create();

            if (useRegex) {
                params.put("$regexp", utf8(toRegularExpression(text)));
            }
            return table.queryPage(params, pageOpts, opts ->
                queryText.query(useRegex ? "paged_find_all" : "paged_find_all_lite", paging(opts))
            );
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Cluster>> partialUpdate(Cluster cluster) {
        try {
            String query = queryText.query("update_partial");
            return table.updateOne(query, cluster);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String clusterId) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$id", utf8(clusterId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteByProjectId(String projectId, String folderId) {
        try {
            String query = queryText.query("delete_for_project");
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String projectId, String folderId, String clusterId) {
        try {
            String query = queryText.query("exists");
            Params params = Params.of("$id", utf8(clusterId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * CLUSTERS TABLE
     */
    private static final class ClustersTable extends YdbTable<String, Cluster> {
        private final ObjectMapper objectMapper;

        ClustersTable(TableClient tableClient, String path, ObjectMapper objectMapper, Executor executor) {
            super(tableClient, path, executor);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("hosts", PrimitiveType.utf8())
                .addNullableColumn("hostUrls", PrimitiveType.utf8())
                .addNullableColumn("conductorGroups", PrimitiveType.utf8())
                .addNullableColumn("conductorTags", PrimitiveType.utf8())
                .addNullableColumn("nannyGroups", PrimitiveType.utf8())
                .addNullableColumn("qloudGroups", PrimitiveType.utf8())
                .addNullableColumn("networks", PrimitiveType.utf8())
                .addNullableColumn("ypClusters", PrimitiveType.utf8())
                .addNullableColumn("instanceGroups", PrimitiveType.utf8())
                .addNullableColumn("cloudDns", PrimitiveType.utf8())
                .addNullableColumn("port", PrimitiveType.int32())
                .addNullableColumn("sensorsTtlDays", PrimitiveType.int32())
                .addNullableColumn("useFqdn", PrimitiveType.bool())
                .addNullableColumn("tvmDestId", PrimitiveType.utf8())
                .addNullableColumn("shardSettings", PrimitiveType.utf8())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected String getId(Cluster cluster) {
            return cluster.getId();
        }

        @Override
        protected Params toParams(Cluster cluster) {
            final ShardSettings settings = cluster.getShardSettings();
            final int port = ShardSettings.getPort(settings, 0);
            final String tvmDestId = settings.getPullSettings() != null ? settings.getPullSettings().getTvmDestinationId() : "";
            final int ttl = settings.getMetricsTtl();
            final boolean useFqdn = settings.getPullSettings() != null && settings.getPullSettings().getHostLabelPolicy() == ShardSettings.HostLabelPolicy.FULL_HOSTNAME;

            return Params.create()
                .put("$id", utf8(cluster.getId()))
                .put("$projectId", utf8(cluster.getProjectId()))
                .put("$folderId", utf8(cluster.getFolderId()))
                .put("$name", utf8(cluster.getName()))
                .put("$description", utf8(cluster.getDescription()))
                .put("$hosts", utf8(toJson(objectMapper, cluster.getHosts())))
                .put("$hostUrls", utf8(toJson(objectMapper, cluster.getHostUrls())))
                .put("$conductorGroups", utf8(toJson(objectMapper, cluster.getConductorGroups())))
                .put("$conductorTags", utf8(toJson(objectMapper, cluster.getConductorTags())))
                .put("$nannyGroups", utf8(toJson(objectMapper, cluster.getNannyGroups())))
                .put("$qloudGroups", utf8(toJson(objectMapper, cluster.getQloudGroups())))
                .put("$networks", utf8(toJson(objectMapper, cluster.getNetworks())))
                .put("$ypClusters", utf8(toJson(objectMapper, cluster.getYpClusters())))
                .put("$instanceGroups", utf8(toJson(objectMapper, cluster.getInstanceGroups())))
                .put("$cloudDns", utf8(toJson(objectMapper, cluster.getCloudDns())))
                .put("$port", int32(port))
                .put("$sensorsTtlDays", int32(ttl))
                .put("$useFqdn", bool(useFqdn))
                .put("$tvmDestId", utf8(tvmDestId))
                .put("$shardSettings", utf8(Nullables.orEmpty(toJson(objectMapper, settings))))
                .put("$labels", utf8(Nullables.orEmpty(toJson(objectMapper, cluster.getLabels()))))
                .put("$createdAt", int64(cluster.getCreatedAtMillis()))
                .put("$updatedAt", int64(cluster.getUpdatedAtMillis()))
                .put("$createdBy", utf8(cluster.getCreatedBy()))
                .put("$updatedBy", utf8(cluster.getUpdatedBy()))
                .put("$version", int32(cluster.getVersion()));
        }

        @Override
        protected Cluster mapFull(ResultSetReader resultSet) {
            return partialCluster(Cluster.newBuilder(), resultSet)
                .setDescription(resultSet.getColumn("description").getUtf8())
                .setLabels(fromJsonMap(objectMapper, resultSet.getColumn("labels").getUtf8()))
                .setCreatedBy(resultSet.getColumn("createdBy").getUtf8())
                .setUpdatedBy(resultSet.getColumn("updatedBy").getUtf8())
                .setVersion(resultSet.getColumn("version").getInt32())
                .build();
        }

        @Override
        protected Cluster mapPartial(ResultSetReader resultSet) {
            return partialCluster(Cluster.newBuilder(), resultSet).build();
        }

        private Cluster.Builder partialCluster(Cluster.Builder builder, ResultSetReader resultSet) {
            var clusterBuilder = builder
                .setId(resultSet.getColumn("id").getUtf8())
                .setCreatedAtMillis(resultSet.getColumn("createdAt").getInt64())
                .setUpdatedAtMillis(resultSet.getColumn("updatedAt").getInt64())
                .setProjectId(resultSet.getColumn("projectId").getUtf8())
                .setFolderId(resultSet.getColumn("folderId").getUtf8())
                .setName(resultSet.getColumn("name").getUtf8())
                .setHosts(fromJson(resultSet, "hosts", ClusterHostListConf.class))
                .setHostUrls(fromJson(resultSet, "hostUrls", ClusterHostUrlConf.class))
                .setConductorGroups(fromJson(resultSet, "conductorGroups", ClusterConductorGroupConf.class))
                .setConductorTags(fromJson(resultSet, "conductorTags", ClusterConductorTagConf.class))
                .setNannyGroups(fromJson(resultSet, "nannyGroups", ClusterNannyGroupConf.class))
                .setQloudGroups(fromJson(resultSet, "qloudGroups", ClusterQloudGroupConf.class))
                .setNetworks(fromJson(resultSet, "networks", ClusterNetworkConf.class))
                .setYpClusters(fromJson(resultSet, "ypClusters", ClusterYpConf.class))
                .setInstanceGroups(fromJson(resultSet, "instanceGroups", ClusterInstanceGroupConf.class))
                .setCloudDns(fromJson(resultSet, "cloudDns", ClusterCloudDnsConf.class));
            return clusterBuilder.setShardSettings(getShardSettings(resultSet, clusterBuilder));
        }

        private <T> List<T> fromJson(ResultSetReader resultSet, String columnName, Class<T> clazz) {
            String json = resultSet.getColumn(columnName).getUtf8();
            return fromJsonList(objectMapper, json, clazz);
        }

        private ShardSettings getShardSettings(ResultSetReader resultSet, Cluster.Builder clusterBuilder) {
            var shardSettingsPersisted = fromJsonOrNull(objectMapper, resultSet.getColumn("shardSettings").getUtf8(), ShardSettings.class);
            final ShardSettings.Type type;
            final ShardSettings.PullSettings pullSettings;
            int port = ShardSettings.getPort(shardSettingsPersisted, 0);
            int ttl = ShardSettings.getMetricsTtlDays(shardSettingsPersisted, 0);
            ShardSettings.HostLabelPolicy policy = ShardSettings.getHostLabelPolicy(shardSettingsPersisted, ShardSettings.HostLabelPolicy.SHORT_HOSTNAME);
            String tvmId = ShardSettings.getTvmDestinationId(shardSettingsPersisted, "");

            boolean isPull = (clusterBuilder.getHosts() != null && clusterBuilder.getHosts().size() > 0) ||
                    (clusterBuilder.getHostUrls() != null && clusterBuilder.getHostUrls().size() > 0) ||
                    (clusterBuilder.getConductorGroups() != null && clusterBuilder.getConductorGroups().size() > 0) ||
                    (clusterBuilder.getConductorTags() != null && clusterBuilder.getConductorTags().size() > 0) ||
                    (clusterBuilder.getNannyGroups() != null && clusterBuilder.getNannyGroups().size() > 0) ||
                    (clusterBuilder.getQloudGroups() != null && clusterBuilder.getQloudGroups().size() > 0) ||
                    (clusterBuilder.getNetworks() != null && clusterBuilder.getNetworks().size() > 0) ||
                    (clusterBuilder.getYpClusters() != null && clusterBuilder.getYpClusters().size() > 0) ||
                    (clusterBuilder.getInstanceGroups() != null && clusterBuilder.getInstanceGroups().size() > 0) ||
                    (clusterBuilder.getCloudDns() != null && clusterBuilder.getCloudDns().size() > 0) ||
                    policy == ShardSettings.HostLabelPolicy.FULL_HOSTNAME ||
                    port > 0 ||
                    !Nullables.orEmpty(tvmId).isEmpty();

            if (!isPull) {
                type = ShardSettings.Type.UNSPECIFIED;
                pullSettings = null;
            } else {
                type = ShardSettings.Type.PULL;
                pullSettings = ShardSettings.PullSettings.newBuilder()
                        .setPort(port)
                        .setPath(ShardSettings.getPath(shardSettingsPersisted, ""))
                        .setAddTsArgs(shardSettingsPersisted != null && shardSettingsPersisted.getPullSettings() != null && shardSettingsPersisted.getPullSettings().isAddTsArgs())
                        .setProtocol(ShardSettings.getProtocol(shardSettingsPersisted, ShardSettings.PullProtocol.UNKNOWN))
                        .setTvmDestinationId(tvmId)
                        .setHostLabelPolicy(policy)
                        .build();
            }
            return ShardSettings.of(
                    type,
                    pullSettings,
                    shardSettingsPersisted != null ? shardSettingsPersisted.getGrid() : 0,
                    ttl,
                    ShardSettings.getDecimPolicy(shardSettingsPersisted, DecimPolicy.UNDEFINED),
                    shardSettingsPersisted != null ? shardSettingsPersisted.getAggregationSettings() : ShardSettings.AggregationSettings.EMPTY,
                    ShardSettings.getInterval(shardSettingsPersisted, 0)
            );
        }
    }
}
