package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.db.dao.DashboardsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.DashboardRow;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static com.yandex.ydb.table.values.PrimitiveValue.float64;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonArray;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;


/**
 * @author Sergey Polovko
 */
public class YdbDashboardsDao implements DashboardsDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("dashboard", Arrays.asList(
        "insert",
        "find_with_project",
        "paged_find",
        "paged_find_all",
        "update_partial",
        "delete_with_project",
        "exists_with_project",
        "find_by_project_shorted",
        "delete_for_project"
    ));

    private final DashboardsTable table;
    private final QueryText queryText;

    public YdbDashboardsDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new DashboardsTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("dashboard.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Boolean> insert(Dashboard dashboard) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, dashboard);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Dashboard>> findOne(String projectId, String folderId, String dashboardId) {
        try {
            String query = queryText.query("find_with_project");
            Params params = Params.of("$id", utf8(dashboardId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<PagedResult<Dashboard>> findByProjectId(
            String projectId, String folderId, PageOptions pageOpts, String text)
    {
        try {
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId), "$regexp", utf8(toRegularExpression(text)));
            return table.queryPage(params, pageOpts, opts -> {
                return queryText.query("paged_find", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<PagedResult<Dashboard>> findAll(PageOptions pageOpts, String text) {
        if (!pageOpts.isLimited() && StringUtils.isBlank(text)) {
            return table.queryAll()
                .thenApply(result -> PagedResult.of(result, pageOpts, result.size()));
        }

        try {
            Params params = Params.of("$regexp", utf8(toRegularExpression(text)));
            return table.queryPage(params, pageOpts, (opts) -> {
                return queryText.query("paged_find_all", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Dashboard>> partialUpdate(Dashboard dashboard) {
        try {
            String query = queryText.query("update_partial");
            return table.updateOne(query, dashboard);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String dashboardId) {
        try {
            String query = queryText.query("delete_with_project");
            Params params = Params.of("$id", utf8(dashboardId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String projectId, String folderId, String dashboardId) {
        try {
            String query = queryText.query("exists_with_project");
            Params params = Params.of("$id", utf8(dashboardId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Dashboard>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<Void> deleteByProjectId(String projectId, String folderId) {
        try {
            String query = queryText.query("delete_for_project");
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<ShortGraphOrDashboardConf>> findByProjectIdShorted(String projectId, String folderId) {
        try {
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryPage(params, PageOptions.MAX, opts -> {
                    return queryText.query("find_by_project_shorted", paging(opts));
                })
                .thenApply(r -> r.getResult()
                    .stream()
                    .map(d -> new ShortGraphOrDashboardConf(d.getId(), d.getProjectId(), d.getName(), d.getParameters()))
                    .collect(Collectors.toList()));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * GRAPHS TABLE
     */
    private static final class DashboardsTable extends YdbTable<String, Dashboard> {
        private final ObjectMapper objectMapper;

        DashboardsTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("heightMultiplier", PrimitiveType.float64())
                .addNullableColumn("parameters", PrimitiveType.utf8())
                .addNullableColumn("rows", PrimitiveType.utf8())
                .addNullableColumn("generatedId", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected String getId(Dashboard dashboard) {
            return dashboard.getId();
        }

        @Override
        protected Params toParams(Dashboard dashboard) {
            return Params.create()
                .put("$id", utf8(dashboard.getId()))
                .put("$projectId", utf8(dashboard.getProjectId()))
                .put("$folderId", utf8(dashboard.getFolderId()))
                .put("$name", utf8(dashboard.getName()))
                .put("$description", utf8(dashboard.getDescription()))
                .put("$heightMultiplier", float64(dashboard.getHeightMultiplier()))
                .put("$parameters", utf8(toJson(objectMapper, dashboard.getParameters())))
                .put("$rows", utf8(toJson(objectMapper, dashboard.getRows())))
                .put("$generatedId", utf8(dashboard.getGeneratedId()))
                .put("$createdAt", int64(dashboard.getCreatedAtMillis()))
                .put("$updatedAt", int64(dashboard.getUpdatedAtMillis()))
                .put("$createdBy", utf8(dashboard.getCreatedBy()))
                .put("$updatedBy", utf8(dashboard.getUpdatedBy()))
                .put("$version", int32(dashboard.getVersion()));
        }

        @Override
        protected Dashboard mapFull(ResultSetReader r) {
            return toPartialDashboard(r, Dashboard.newBuilder())
                .setDescription(r.getColumn("description").getUtf8())
                .setHeightMultiplier(r.getColumn("heightMultiplier").getFloat64())
                .setRows(fromJsonArray(objectMapper, r.getColumn("rows").getUtf8(), DashboardRow.class))
                .setGeneratedId(r.getColumn("generatedId").getUtf8())
                .setCreatedAtMillis(r.getColumn("createdAt").getInt64())
                .setCreatedBy(r.getColumn("createdBy").getUtf8())
                .setUpdatedAtMillis(r.getColumn("updatedAt").getInt64())
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                .setVersion(r.getColumn("version").getInt32())
                .build();
        }

        @Override
        protected Dashboard mapPartial(ResultSetReader r) {
            return toPartialDashboard(r, Dashboard.newBuilder()).build();
        }

        private Dashboard.Builder toPartialDashboard(ResultSetReader r, Dashboard.Builder builder) {
            int parametersIdx = r.getColumnIndex("parameters");
            if (parametersIdx != -1) {
                String json = r.getColumn(parametersIdx).getUtf8();
                builder.setParameters(fromJsonArray(objectMapper, json, Selector.class));
            }
            return builder
                .setId(r.getColumn("id").getUtf8())
                .setProjectId(r.getColumn("projectId").getUtf8())
                .setFolderId(r.getColumn("folderId").getUtf8())
                .setName(r.getColumn("name").getUtf8());
        }
    }
}
