package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.db.dao.EntityFlagsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbEntityFlagsDao implements EntityFlagsDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate("entity_flags", Arrays.asList(
            "upsert",
            "delete"
    ));

    private final Table table;
    private final QueryText queryText;

    public YdbEntityFlagsDao(TableClient client, String path) {
        this.table = new Table(client, path);
        this.queryText = TEMPLATE.build(Collections.singletonMap("entity_flags.table.path", path));
    }

    @Override
    public CompletableFuture<List<Record>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<Void> upsert(EntityFlagsDao.Record record) {
        try {
            String query = queryText.query("upsert");
            return table.upsertOne(query, record);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteOne(String flag, String projectId, String id) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$flag", utf8(flag), "$projectId", utf8(projectId), "$id", utf8(id));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static final class Table extends YdbTable<String, Record> {
        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("flag", PrimitiveType.utf8())
                    .addNullableColumn("projectId", PrimitiveType.utf8())
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .addNullableColumn("value", PrimitiveType.bool())
                    .setPrimaryKeys("flag", "projectId", "id")
                    .build();
        }

        @Override
        protected String getId(Record record) {
            return record.flag + " " + record.projectId + " " + record.id;
        }

        @Override
        protected Params toParams(Record record) {
            return Params.create()
                    .put("$flag", utf8(record.flag))
                    .put("$projectId", utf8(record.projectId))
                    .put("$id", utf8(record.id))
                    .put("$value", bool(record.value));
        }

        @Override
        protected Record mapFull(ResultSetReader r) {
            String flag = r.getColumn("flag").getUtf8();
            String projectId = r.getColumn("projectId").getUtf8();
            String id = r.getColumn("id").getUtf8();
            boolean value = r.getColumn("value").getBool();
            return new Record(flag, projectId, id, value);
        }

        @Override
        protected Record mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
