package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.db.dao.FolderMenuDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.FolderMenu;
import ru.yandex.solomon.core.db.model.MenuItem;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonArray;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;


/**
 * @author Sergey Polovko
 */
public class YdbFolderMenuDao implements FolderMenuDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("folder_menu", Arrays.asList(
            "find",
            "upsert",
            "delete"
    ));

    private final FolderMenuTable table;
    private final QueryText queryText;

    public YdbFolderMenuDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new FolderMenuTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("folder.menu.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Optional<FolderMenu>> findById(String projectId, String id) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$projectId", utf8(projectId), "$id", utf8(id));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> insert(FolderMenu menu) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, menu);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<FolderMenu>> upsert(FolderMenu folderMenu) {
        try {
            String query = queryText.query("upsert");
            return table.updateOne(query, folderMenu)
                    .thenApply(newMenu -> {
                        final int oldVersion = folderMenu.getVersion();
                        if (newMenu.isPresent()) {
                            return newMenu.filter(m -> m.getVersion() == oldVersion);
                        }
                        return Optional.of(folderMenu.toBuilder().setVersion(oldVersion + 1).build());
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteById(String projectId, String id) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$projectId", utf8(projectId), "$id", utf8(id));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * FOLDER MENU TABLE
     */
    private static final class FolderMenuTable extends YdbTable<String, FolderMenu> {

        private final ObjectMapper objectMapper;

        FolderMenuTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .addNullableColumn("projectId", PrimitiveType.utf8())
                    .addNullableColumn("items", PrimitiveType.utf8())
                    .addNullableColumn("createdAt", PrimitiveType.int64())
                    .addNullableColumn("updatedAt", PrimitiveType.int64())
                    .addNullableColumn("createdBy", PrimitiveType.utf8())
                    .addNullableColumn("updatedBy", PrimitiveType.utf8())
                    .addNullableColumn("version", PrimitiveType.int32())
                    .setPrimaryKeys("projectId", "id")
                    .build();
        }

        @Override
        protected String getId(FolderMenu menu) {
            return menu.getId();
        }

        @Override
        protected Params toParams(FolderMenu menu) {
            return Params.create()
                    .put("$id", utf8(menu.getId()))
                    .put("$projectId", utf8(menu.getProjectId()))
                    .put("$items", utf8(toJson(objectMapper, menu.getItems())))
                    .put("$createdAt", int64(menu.getCreatedAtMillis()))
                    .put("$updatedAt", int64(menu.getUpdatedAtMillis()))
                    .put("$createdBy", utf8(menu.getCreatedBy()))
                    .put("$updatedBy", utf8(menu.getUpdatedBy()))
                    .put("$version", int32(menu.getVersion()));
        }

        @Override
        protected FolderMenu mapFull(ResultSetReader r) {
            return FolderMenu.newBuilder()
                    .setId(r.getColumn("id").getUtf8())
                    .setProjectId(r.getColumn("projectId").getUtf8())
                    .setItems(fromJsonArray(objectMapper, r.getColumn("items").getUtf8(), MenuItem.class))
                    .setCreatedAtMillis(r.getColumn("createdAt").getInt64())
                    .setCreatedBy(r.getColumn("createdBy").getUtf8())
                    .setUpdatedAtMillis(r.getColumn("updatedAt").getInt64())
                    .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                    .setVersion(r.getColumn("version").getInt32())
                    .build();
        }

        @Override
        protected FolderMenu mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
