package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.db.dao.GraphsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.core.db.model.graph.AggregationType;
import ru.yandex.solomon.core.db.model.graph.ColorSchemeType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingAggregationType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingFillType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingMode;
import ru.yandex.solomon.core.db.model.graph.FilterFunction;
import ru.yandex.solomon.core.db.model.graph.FilterOrder;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.core.db.model.graph.GraphElement;
import ru.yandex.solomon.core.db.model.graph.GraphMode;
import ru.yandex.solomon.core.db.model.graph.GraphTransform;
import ru.yandex.solomon.core.db.model.graph.InterpolationType;
import ru.yandex.solomon.core.db.model.graph.OverLinesTransform;
import ru.yandex.solomon.core.db.model.graph.ScaleType;
import ru.yandex.solomon.core.db.model.graph.SecondaryGraphMode;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.float64;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonArray;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;


/**
 * @author Sergey Polovko
 */
public class YdbGraphsDao implements GraphsDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("graph", Arrays.asList(
        "insert",
        "find_with_project",
        "paged_find",
        "paged_find_all",
        "update_partial",
        "delete_with_project",
        "exists_with_project",
        "find_by_project_shorted",
        "delete_for_project"
    ));

    private final GraphsTable table;
    private final QueryText queryText;

    public YdbGraphsDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new GraphsTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("graph.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Boolean> insert(Graph graph) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, graph);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Graph>> findOne(String projectId, String folderId, String graphId) {
        try {
            String query = queryText.query("find_with_project");
            Params params = Params.of("$id", utf8(graphId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<PagedResult<Graph>> findByProjectId(
        String projectId, String folderId, PageOptions pageOpts, String text)
    {
        try {
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId), "$regexp", utf8(toRegularExpression(text)));
            return table.queryPage(params, pageOpts, opts -> {
                return queryText.query("paged_find", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<PagedResult<Graph>> findAll(PageOptions pageOpts, String text) {
        if (!pageOpts.isLimited() && StringUtils.isBlank(text)) {
            return table.queryAll()
                .thenApply(result -> PagedResult.of(result, pageOpts, result.size()));
        }

        try {
            Params params = Params.of("$regexp", utf8(toRegularExpression(text)));
            return table.queryPage(params, pageOpts, (opts) -> {
                return queryText.query("paged_find_all", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Graph>> partialUpdate(Graph graph) {
        try {
            String query = queryText.query("update_partial");
            return table.updateOne(query, graph);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String graphId) {
        try {
            String query = queryText.query("delete_with_project");
            Params params = Params.of("$id", utf8(graphId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String projectId, String folderId, String graphId) {
        try {
            String query = queryText.query("exists_with_project");
            Params params = Params.of("$id", utf8(graphId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Graph>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<Void> deleteByProjectId(String projectId, String folderId) {
        try {
            String query = queryText.query("delete_for_project");
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<ShortGraphOrDashboardConf>> findByProjectIdShorted(String projectId, String folderId) {
        try {
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryPage(params, PageOptions.MAX, opts -> {
                    return queryText.query("find_by_project_shorted", paging(opts));
                })
                .thenApply(r -> r.getResult()
                    .stream()
                    .map(g -> new ShortGraphOrDashboardConf(g.getId(), g.getProjectId(), g.getName(), g.getParameters()))
                    .collect(Collectors.toList()));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * GRAPHS TABLE
     */
    private static final class GraphsTable extends YdbTable<String, Graph> {
        private static final OptionalType OPTIONAL_BOOL = OptionalType.of(PrimitiveType.bool());
        private static final OptionalType OPTIONAL_FLOAT64 = OptionalType.of(PrimitiveType.float64());

        private final ObjectMapper objectMapper;

        GraphsTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("parameters", PrimitiveType.utf8())
                .addNullableColumn("variables", PrimitiveType.utf8())  // TODO: drop column
                .addNullableColumn("elements", PrimitiveType.utf8())
                .addNullableColumn("graphMode", PrimitiveType.utf8())
                .addNullableColumn("secondaryGraphMode", PrimitiveType.utf8())
                .addNullableColumn("min", PrimitiveType.utf8())
                .addNullableColumn("max", PrimitiveType.utf8())
                .addNullableColumn("normalize", PrimitiveType.bool())
                .addNullableColumn("colorScheme", PrimitiveType.utf8())
                .addNullableColumn("dropNans", PrimitiveType.bool())
                .addNullableColumn("stack", PrimitiveType.bool())
                .addNullableColumn("aggr", PrimitiveType.utf8())
                .addNullableColumn("interpolate", PrimitiveType.utf8())
                .addNullableColumn("scale", PrimitiveType.utf8())
                .addNullableColumn("numberFormat", PrimitiveType.utf8())
                .addNullableColumn("green", PrimitiveType.utf8())
                .addNullableColumn("yellow", PrimitiveType.utf8())
                .addNullableColumn("red", PrimitiveType.utf8())
                .addNullableColumn("violet", PrimitiveType.utf8())
                .addNullableColumn("greenValue", PrimitiveType.utf8())
                .addNullableColumn("yellowValue", PrimitiveType.utf8())
                .addNullableColumn("redValue", PrimitiveType.utf8())
                .addNullableColumn("violetValue", PrimitiveType.utf8())
                .addNullableColumn("sortByLabel", PrimitiveType.utf8())
                .addNullableColumn("asc", PrimitiveType.bool())
                .addNullableColumn("limit", PrimitiveType.utf8())
                .addNullableColumn("overLinesTransform", PrimitiveType.utf8())
                .addNullableColumn("percentiles", PrimitiveType.utf8())
                .addNullableColumn("bucketLabel", PrimitiveType.utf8())
                .addNullableColumn("ignoreInf", PrimitiveType.bool())
                .addNullableColumn("filter", PrimitiveType.utf8())
                .addNullableColumn("filterBy", PrimitiveType.utf8())
                .addNullableColumn("filterLimit", PrimitiveType.utf8())
                .addNullableColumn("transform", PrimitiveType.utf8())
                .addNullableColumn("movingWindow", PrimitiveType.utf8())
                .addNullableColumn("movingPercentile", PrimitiveType.utf8())
                .addNullableColumn("downsampling", PrimitiveType.utf8())
                .addNullableColumn("downsamplingAggr", PrimitiveType.utf8())
                .addNullableColumn("downsamplingFill", PrimitiveType.utf8())
                .addNullableColumn("ignoreMinStepMillis", PrimitiveType.bool())
                .addNullableColumn("grid", PrimitiveType.utf8())
                .addNullableColumn("maxPoints", PrimitiveType.int32())
                .addNullableColumn("hideNoData", PrimitiveType.bool())
                .addNullableColumn("threshold", PrimitiveType.float64())
                .addNullableColumn("generatedId", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected String getId(Graph graph) {
            return graph.getId();
        }

        @Override
        protected Params toParams(Graph graph) {
            return Params.create()
                .put("$id", utf8(graph.getId()))
                .put("$projectId", utf8(graph.getProjectId()))
                .put("$folderId", utf8(graph.getFolderId()))
                .put("$name", utf8(graph.getName()))
                .put("$description", utf8(graph.getDescription()))
                .put("$parameters", utf8(toJson(objectMapper, graph.getParameters())))
                .put("$elements", utf8(toJson(objectMapper, graph.getElements())))
                .put("$graphMode", utf8(graph.getGraphMode().name()))
                .put("$secondaryGraphMode", utf8(graph.getSecondaryGraphMode().name()))
                .put("$min", utf8(graph.getMin()))
                .put("$max", utf8(graph.getMax()))
                .put("$normalize", bool(graph.isNormalize()))
                .put("$colorScheme", utf8(graph.getColorScheme().name()))
                .put("$dropNans", optionalBool(graph.isDropNans()))
                .put("$stack", optionalBool(graph.getStack()))
                .put("$aggr", utf8(graph.getAggr().name()))
                .put("$interpolate", utf8(graph.getInterpolate().name()))
                .put("$scale", utf8(graph.getScale().name()))
                .put("$numberFormat", utf8(graph.getNumberFormat()))
                .put("$green", utf8(graph.getGreen()))
                .put("$yellow", utf8(graph.getYellow()))
                .put("$red", utf8(graph.getRed()))
                .put("$violet", utf8(graph.getViolet()))
                .put("$greenValue", utf8(graph.getGreenValue()))
                .put("$yellowValue", utf8(graph.getYellowValue()))
                .put("$redValue", utf8(graph.getRedValue()))
                .put("$violetValue", utf8(graph.getVioletValue()))
                .put("$sortByLabel", utf8(graph.getSortByLabel()))
                .put("$asc", bool(graph.isAsc()))
                .put("$limit", utf8(graph.getLimit()))
                .put("$overLinesTransform", utf8(graph.getOverLinesTransform().name()))
                .put("$percentiles", utf8(graph.getPercentiles()))
                .put("$bucketLabel", utf8(graph.getBucketLabel()))
                .put("$ignoreInf", bool(graph.isIgnoreInf()))
                .put("$filter", utf8(graph.getFilter().name()))
                .put("$filterBy", utf8(graph.getFilterBy().name()))
                .put("$filterLimit", utf8(graph.getFilterLimit()))
                .put("$transform", utf8(graph.getTransform().name()))
                .put("$movingWindow", utf8(graph.getMovingWindow()))
                .put("$movingPercentile", utf8(graph.getMovingPercentile()))
                .put("$downsampling", utf8(graph.getDownsampling().name()))
                .put("$downsamplingAggr", utf8(graph.getDownsamplingAggr().name()))
                .put("$downsamplingFill", utf8(graph.getDownsamplingFill().name()))
                .put("$ignoreMinStepMillis", bool(graph.isIgnoreMinStepMillis()))
                .put("$grid", utf8(graph.getGrid()))
                .put("$maxPoints", int32(graph.getMaxPoints()))
                .put("$hideNoData", bool(graph.isHideNoData()))
                .put("$threshold", optionalFloat64(graph.getThreshold()))
                .put("$generatedId", utf8(graph.getGeneratedId()))
                .put("$createdAt", int64(graph.getCreatedAtMillis()))
                .put("$updatedAt", int64(graph.getUpdatedAtMillis()))
                .put("$createdBy", utf8(graph.getCreatedBy()))
                .put("$updatedBy", utf8(graph.getUpdatedBy()))
                .put("$version", int32(graph.getVersion()));
        }

        @Override
        protected Graph mapFull(ResultSetReader r) {
            int bucketLabelIndex = r.getColumnIndex("bucketLabel");
            int downsamplingFillIndex = r.getColumnIndex("downsamplingFill");
            int ignoreMinStepMillisIndex = r.getColumnIndex("ignoreMinStepMillis");

            DownsamplingFillType downsamplingFill = DownsamplingFillType.DEFAULT_VALUE;
            if (downsamplingFillIndex >= 0) {
                String downsamplingFillStr = r.getColumn(downsamplingFillIndex).getUtf8();
                if (!StringUtils.isBlank(downsamplingFillStr)) {
                    downsamplingFill = DownsamplingFillType.valueOf(downsamplingFillStr);
                }
            }

            Graph.Builder builder = toPartialGraph(Graph.newBuilder(), r)
                .setDescription(r.getColumn("description").getUtf8())
                .setElements(fromJsonArray(objectMapper, r.getColumn("elements").getUtf8(), GraphElement.class))
                .setGraphMode(GraphMode.valueOf(r.getColumn("graphMode").getUtf8()))
                .setSecondaryGraphMode(SecondaryGraphMode.valueOf(r.getColumn("secondaryGraphMode").getUtf8()))
                .setMin(r.getColumn("min").getUtf8())
                .setMax(r.getColumn("max").getUtf8())
                .setNormalize(r.getColumn("normalize").getBool())
                .setColorScheme(ColorSchemeType.valueOf(r.getColumn("colorScheme").getUtf8()))
                .setAggr(AggregationType.valueOf(r.getColumn("aggr").getUtf8()))
                .setInterpolate(InterpolationType.valueOf(r.getColumn("interpolate").getUtf8()))
                .setScale(ScaleType.valueOf(r.getColumn("scale").getUtf8()))
                .setNumberFormat(r.getColumn("numberFormat").getUtf8())
                .setGreen(r.getColumn("green").getUtf8())
                .setYellow(r.getColumn("yellow").getUtf8())
                .setRed(r.getColumn("red").getUtf8())
                .setViolet(r.getColumn("violet").getUtf8())
                .setGreenValue(r.getColumn("greenValue").getUtf8())
                .setYellowValue(r.getColumn("yellowValue").getUtf8())
                .setRedValue(r.getColumn("redValue").getUtf8())
                .setVioletValue(r.getColumn("violetValue").getUtf8())
                .setSortByLabel(r.getColumn("sortByLabel").getUtf8())
                .setIgnoreInf(r.getColumn("ignoreInf").getBool())
                .setAsc(r.getColumn("asc").getBool())
                .setLimit(r.getColumn("limit").getUtf8())
                .setOverLinesTransform(OverLinesTransform.valueOf(r.getColumn("overLinesTransform").getUtf8()))
                .setPercentiles(r.getColumn("percentiles").getUtf8())
                .setBucketLabel(bucketLabelIndex >= 0 ? r.getColumn(bucketLabelIndex).getUtf8() : "")
                .setFilter(FilterOrder.valueOf(r.getColumn("filter").getUtf8()))
                .setFilterBy(FilterFunction.valueOf(r.getColumn("filterBy").getUtf8()))
                .setFilterLimit(r.getColumn("filterLimit").getUtf8())
                .setTransform(GraphTransform.valueOf(r.getColumn("transform").getUtf8()))
                .setMovingWindow(r.getColumn("movingWindow").getUtf8())
                .setMovingPercentile(r.getColumn("movingPercentile").getUtf8())
                .setDownsampling(DownsamplingMode.valueOf(r.getColumn("downsampling").getUtf8()))
                .setDownsamplingAggr(DownsamplingAggregationType.valueOf(r.getColumn("downsamplingAggr").getUtf8()))
                .setDownsamplingFill(downsamplingFill)
                .setIgnoreMinStepMillis(ignoreMinStepMillisIndex >= 0 && r.getColumn(ignoreMinStepMillisIndex).getBool())
                .setGrid(r.getColumn("grid").getUtf8())
                .setMaxPoints(r.getColumn("maxPoints").getInt32())
                .setHideNoData(r.getColumn("hideNoData").getBool())
                .setGeneratedId(r.getColumn("generatedId").getUtf8())
                .setCreatedAtMillis(r.getColumn("createdAt").getInt64())
                .setCreatedBy(r.getColumn("createdBy").getUtf8())
                .setUpdatedAtMillis(r.getColumn("updatedAt").getInt64())
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                .setVersion(r.getColumn("version").getInt32());

            ValueReader thresholdColumn = r.getColumn("threshold");
            if (thresholdColumn.isOptionalItemPresent()) {
                builder.setThreshold(thresholdColumn.getFloat64());
            }
            ValueReader dropNansColumn = r.getColumn("dropNans");
            if (dropNansColumn.isOptionalItemPresent()) {
                builder.setDropNans(dropNansColumn.getBool());
            }
            ValueReader stackColumn = r.getColumn("stack");
            if (stackColumn.isOptionalItemPresent()) {
                builder.setStack(stackColumn.getBool());
            }
            return builder.build();
        }

        @Override
        protected Graph mapPartial(ResultSetReader r) {
            return toPartialGraph(Graph.newBuilder(), r).build();
        }

        private Graph.Builder toPartialGraph(Graph.Builder builder, ResultSetReader r) {
            int parametersIdx = r.getColumnIndex("parameters");
            if (parametersIdx != -1) {
                String json = r.getColumn(parametersIdx).getUtf8();
                builder.setParameters(fromJsonArray(objectMapper, json, Selector.class));
            }

            return builder
                .setId(r.getColumn("id").getUtf8())
                .setName(r.getColumn("name").getUtf8())
                .setProjectId(r.getColumn("projectId").getUtf8())
                .setFolderId(r.getColumn("folderId").getUtf8());
        }

        private static OptionalValue optionalBool(@Nullable Boolean value) {
            if (value == null) {
                return OPTIONAL_BOOL.emptyValue();
            }
            return OPTIONAL_BOOL.newValue(bool(value));
        }

        private static OptionalValue optionalFloat64(@Nullable Double value) {
            if (value == null) {
                return OPTIONAL_FLOAT64.emptyValue();
            }
            return OPTIONAL_FLOAT64.newValue(float64(value));
        }
    }
}
