package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.db.dao.ProjectMenuDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.MenuItem;
import ru.yandex.solomon.core.db.model.ProjectMenu;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonArray;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;


/**
 * @author Sergey Polovko
 */
public class YdbProjectMenuDao implements ProjectMenuDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("menu_project", Arrays.asList(
        "find",
        "insert",
        "upsert",
        "delete"
    ));

    private final ProjectMenuTable table;
    private final QueryText queryText;

    public YdbProjectMenuDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new ProjectMenuTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("project.menu.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Optional<ProjectMenu>> findById(String id) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$id", utf8(id));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> insert(ProjectMenu projectMenu) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, projectMenu);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<ProjectMenu>> upsert(ProjectMenu projectMenu) {
        try {
            String query = queryText.query("upsert");
            return table.updateOne(query, projectMenu)
                .thenApply(newMenu -> {
                    final int oldVersion = projectMenu.getVersion();
                    if (newMenu.isPresent()) {
                        return newMenu.filter(m -> m.getVersion() == oldVersion);
                    }
                    return Optional.of(projectMenu.toBuilder().setVersion(oldVersion + 1).build());
                });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteById(String id) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$id", utf8(id));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * PROJECT MENU TABLE
     */
    private static final class ProjectMenuTable extends YdbTable<String, ProjectMenu> {

        private final ObjectMapper objectMapper;

        ProjectMenuTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("items", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKey("id")
                .build();
        }

        @Override
        protected String getId(ProjectMenu projectMenu) {
            return projectMenu.getId();
        }

        @Override
        protected Params toParams(ProjectMenu menu) {
            return Params.create()
                .put("$id", utf8(menu.getId()))
                .put("$items", utf8(toJson(objectMapper, menu.getItems())))
                .put("$createdAt", int64(menu.getCreatedAtMillis()))
                .put("$updatedAt", int64(menu.getUpdatedAtMillis()))
                .put("$createdBy", utf8(menu.getCreatedBy()))
                .put("$updatedBy", utf8(menu.getUpdatedBy()))
                .put("$version", int32(menu.getVersion()));
        }

        @Override
        protected ProjectMenu mapFull(ResultSetReader r) {
            return ProjectMenu.newBuilder()
                .setId(r.getColumn("id").getUtf8())
                .setItems(fromJsonArray(objectMapper, r.getColumn("items").getUtf8(), MenuItem.class))
                .setCreatedAtMillis(r.getColumn("createdAt").getInt64())
                .setCreatedBy(r.getColumn("createdBy").getUtf8())
                .setUpdatedAtMillis(r.getColumn("updatedAt").getInt64())
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                .setVersion(r.getColumn("version").getInt32())
                .build();
        }

        @Override
        protected ProjectMenu mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
