package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.db.dao.ProjectSettingsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.ProjectSettings;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonMap;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class YdbProjectSettingsDao implements ProjectSettingsDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("project_settings", Arrays.asList(
            "find",
            "upsert",
            "delete"
    ));

    private final ProjectSettingsTable table;
    private final QueryText queryText;

    public YdbProjectSettingsDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new ProjectSettingsTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("project.settings.table.path", tablePath));
    }

    @Override
    public CompletableFuture<List<ProjectSettings>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<ProjectSettings> find(String id) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$id", utf8(id));
            return table.queryOne(query, params)
                    .thenApply(newSettings -> newSettings.orElseGet(() -> new ProjectSettings(id, Collections.emptyMap())));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<ProjectSettings> upsert(ProjectSettings settings) {
        try {
            String query = queryText.query("upsert");
            return table.updateOne(query, settings)
                    .thenApply(ignore -> settings);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(String id) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$id", utf8(id));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static final class ProjectSettingsTable extends YdbTable<String, ProjectSettings> {

        private final ObjectMapper objectMapper;

        ProjectSettingsTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .addNullableColumn("settings", PrimitiveType.utf8())
                    .setPrimaryKey("id")
                    .build();
        }

        @Override
        protected String getId(ProjectSettings s) {
            return s.getId();
        }

        @Override
        protected Params toParams(ProjectSettings s) {
            return Params.create()
                    .put("$id", utf8(s.getId()))
                    .put("$settings", utf8(toJson(objectMapper, s.getSettings())));
        }

        @Override
        protected ProjectSettings mapFull(ResultSetReader r) {
            String id = r.getColumn("id").getUtf8();
            Map<String, String> settings = fromJsonMap(objectMapper, r.getColumn("settings").getUtf8());
            return new ProjectSettings(id, settings);
        }

        @Override
        protected ProjectSettings mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
