package ru.yandex.solomon.core.db.dao.ydb;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Value;

import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.text.TextWithNumbersComparator;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PageTokenCodec;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonMap;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.setFromTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsvRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.pagingToken;


/**
 * @author Sergey Polovko
 */
public class YdbProjectsDao implements ProjectsDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("project", Arrays.asList(
        "exists",
        "find",
        "paged_find_with_text",
        "paged_find_without_text",
        "paged_find_with_text_in_projects",
        "paged_find_without_text_in_projects",
        "paged_find_v3",
        "paged_list",
        "insert",
        "update_partial",
        "delete"
    ));

    private static final StructType ID = StructType.of("id", PrimitiveType.utf8());
    private static final ListType LIST_ID = ListType.of(ID);

    // only mandatory fields
    private static final StructType LITE_PROJECT_TYPE = StructType.of(Map.of(
            "id", PrimitiveType.utf8(),
            "name", PrimitiveType.utf8(),
            "owner", PrimitiveType.utf8(),
            "onlyAuthPush", PrimitiveType.bool()
    ));
    private static final ListType LITE_PROJECT_LIST_TYPE = ListType.of(LITE_PROJECT_TYPE);

    private final ProjectsTable table;
    private final QueryText queryText;
    private final String upsertQuery;

    public YdbProjectsDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper, Executor executor) {
        this.table = new ProjectsTable(tableClient, tablePath, objectMapper, executor);
        this.queryText = TEMPLATE.build(Map.of("project.table.path", tablePath, "project.id.table", LIST_ID));
        this.upsertQuery = String.format("""
                --!syntax_v1
                DECLARE $rows AS %s;
                UPSERT INTO `%s` SELECT * FROM AS_TABLE($rows);
                """, LITE_PROJECT_LIST_TYPE, tablePath);
    }

    @Override
    public CompletableFuture<Boolean> insert(Project project) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, project);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Project>> findById(String id) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$id", utf8(id));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Project>> findAllNames() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<PagedResult<Project>> find(
        String text,
        String abcFilter,
        String login,
        @Nullable EnumSet<ProjectPermission> filterByPermissions,
        PageOptions pageOpts)
    {
        try {
            if (!pageOpts.isLimited()) {
                return table.queryAll().thenApply(projects -> {
                    List<Project> filtered = filterProjectsByOptions(projects, text, abcFilter, login, filterByPermissions);
                    return PagedResult.of(filtered, PageOptions.ALL, filtered.size());
                });
            }

            boolean hasPermissions = filterByPermissions != null && !filterByPermissions.isEmpty();
            // We don't support old permissions (UPDATE, DELETE) here
            Params params = Params.create()
                .put("$login", utf8(login))
                .put("$loginRegexp", utf8(toTsvRegularExpression(login)))
                .put("$hasPermissions", bool(hasPermissions))
                .put("$canRead", bool(hasPermissions && filterByPermissions.contains(ProjectPermission.READ)))
                .put("$canUpdate", bool(hasPermissions && filterByPermissions.contains(ProjectPermission.CONFIG_UPDATE)))
                .put("$canDelete", bool(hasPermissions && filterByPermissions.contains(ProjectPermission.CONFIG_DELETE)))
                .put("$canWrite", bool(hasPermissions && filterByPermissions.contains(ProjectPermission.WRITE)))
                .put("$abcFilter", utf8(abcFilter));

            boolean hasText = !text.isEmpty();
            if (hasText) {
                params
                    .put("$text", utf8(text))
                    .put("$textRegexp", utf8(toRegularExpression(text)));
            }

            return table.queryPage(params, pageOpts, (opts) -> {
                return queryText.query(hasText ? "paged_find_with_text" : "paged_find_without_text", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Project>> list(TokenPageOptions opts) {
        return safeCall(() -> {
            var pageToken = PageTokenLastId.decode(opts.getPageToken());
            String query = queryText.query("paged_list", pagingToken(opts));
            var params = Params.create()
                    .put("$lastId", utf8(pageToken.lastId()));

            return table.queryPage(
                    query,
                    params,
                    opts,
                    p -> new PageTokenLastId(p.getId()).encode());
        });
    }

    @Override
    public CompletableFuture<PagedResult<Project>> findInProjects(
            String text,
            String abcFilter,
            Set<String> projectIds,
            PageOptions pageOpts)
    {
        if (projectIds.isEmpty()) {
            return CompletableFuture.completedFuture(PagedResult.empty());
        }
        try {
            if (!pageOpts.isLimited()) {
                return table.queryAll().thenApply(projects -> {
                    List<Project> filtered = filterProjectsByOptions(projects, text, abcFilter, projectIds);
                    return PagedResult.of(filtered, PageOptions.ALL, filtered.size());
                });
            }
            List<Value> values = new ArrayList<>(projectIds.size());
            for (var id : projectIds) {
                values.add(ID.newValue("id", utf8(id)));
            }
            Params params = Params.create()
                    .put("$ids", LIST_ID.newValue(values))
                    .put("$abcFilter", utf8(abcFilter));
            boolean hasText = !text.isEmpty();
            if (hasText) {
                params
                    .put("$text", utf8(text))
                    .put("$textRegexp", utf8(toRegularExpression(text)));
            }

            return table.queryPage(params, pageOpts, (opts) -> {
                return queryText.query(hasText
                            ? "paged_find_with_text_in_projects"
                            : "paged_find_without_text_in_projects",
                    paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Project>> findV3(String text, int pageSize, String pageToken) {
        try {
            TokenPageOptions pageOpts = new TokenPageOptions(pageSize, pageToken);

            Params params = Params.create()
                    .put("$text", utf8(text))
                    .put("$textRegexp", utf8(toRegularExpression(text)));

            return table.queryPage(params, pageOpts, (opts) -> queryText.query("paged_find_v3", paging(opts)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Project>> partialUpdate(
        Project project,
        boolean canChangeOwner,
        boolean canChangeInternalOptions,
        boolean canUpdateOldFields)
    {
        try {
            Params params = Params.create()
                .put("$id", utf8(project.getId()))
                .put("$name", utf8(project.getName()))
                .put("$description", utf8(project.getDescription()))
                .put("$owner", utf8(canChangeOwner ? project.getOwner() : ""))
                .put("$abcService", utf8(project.getAbcService()))
                .put("$readAcl", utf8(toTsv(project.getAcl().getCanRead())))
                .put("$updateAcl", utf8(toTsv(project.getAcl().getCanUpdate())))
                .put("$deleteAcl", utf8(toTsv(project.getAcl().getCanDelete())))
                .put("$writeAcl", utf8(toTsv(project.getAcl().getCanWrite())))
                .put("$onlyAuthRead", bool(project.isOnlyAuthRead()))
                .put("$updatedAt", int64(project.getUpdatedAtMillis()))
                .put("$updatedBy", utf8(project.getUpdatedBy()))
                .put("$version", int32(project.getVersion()))
                .put("$onlySensorNameShards", bool(project.isOnlyMetricNameShards()))
                .put("$onlyNewFormatWrites", bool(project.isOnlyNewFormatWrites()))
                .put("$labels", utf8(Nullables.orEmpty(toJson(table.objectMapper, project.getLabels()))))
                .put("$onlyNewFormatReads", bool(project.isOnlyNewFormatReads()))
                .put("$metricNameLabel", utf8(project.getMetricNameLabel()))
                .put("$canUpdateOldFields", bool(canUpdateOldFields));


            if (canChangeInternalOptions || project.isOnlyAuthPush()) {
                params.put("$onlyAuthPush", bool(project.isOnlyAuthPush()).makeOptional());
            }

            String query = queryText.query("update_partial");
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> upsertProjects(List<Project> projects) {
        return findAllNames()
            .thenCompose(exists -> {
                var existsProjects = exists.stream()
                        .collect(Collectors.toMap(Project::getId, p -> p));

                var toUpsert = new ArrayList<Project>();
                for (Project project : projects) {
                    var existProject = existsProjects.get(project.getId());
                    if (existProject == null) {
                        toUpsert.add(project);
                    } else if (!existProject.getName().equals(project.getName())) {
                        toUpsert.add(project);
                    } else if (existProject.isOnlyAuthPush() != project.isOnlyAuthPush()) {
                        toUpsert.add(project);
                    }
                }

                CompletableFuture<Void> future = CompletableFuture.completedFuture(null);
                if (!toUpsert.isEmpty()) {
                    for (List<Project> batch : Lists.partition(toUpsert, 300)) {
                        future = future.thenCompose(aVoid -> updateNamesForBatch(batch));
                    }
                }
                return future;
            });
    }

    private CompletableFuture<Void> updateNamesForBatch(List<Project> projects) {
        List<Value> values = projects.stream()
                .map(p -> LITE_PROJECT_TYPE.newValue(
                        "id", PrimitiveValue.utf8(p.getId()),
                        "name", PrimitiveValue.utf8(p.getName()),
                        "owner", PrimitiveValue.utf8(p.getOwner()),
                        "onlyAuthPush", PrimitiveValue.bool(p.isOnlyAuthPush())
                ))
                .collect(Collectors.toList());

        Params params = Params.of("$rows", LITE_PROJECT_LIST_TYPE.newValue(values));
        return table.queryVoid(upsertQuery, params);
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String id, boolean skipValidation) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$id", utf8(id));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String id) {
        try {
            String query = queryText.query("exists");
            Params params = Params.of("$id", utf8(id));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static List<Project> filterProjectsByOptions(
            List<Project> projects,
            String text,
            String abcFilter,
            String login,
            @Nullable EnumSet<ProjectPermission> filterByPermissions)
    {
        Stream<Project> stream = projects.stream();
        stream = stream.filter(p -> hasPermissionsInProject(p, login, filterByPermissions));
        if (!abcFilter.isEmpty()) {
            stream = stream.filter(project -> abcFilter.equals(project.getAbcService()));
        }
        if (!text.isEmpty()) {
            stream = filterProjectsByText(text, stream);
        }
        return stream
            .sorted(Comparator.comparing(Project::getId, TextWithNumbersComparator.instance))
            .collect(Collectors.toList());
    }

    private static List<Project> filterProjectsByOptions(
            List<Project> projects,
            String text,
            String abcFilter,
            Set<String> projectIds)
    {
        Stream<Project> stream = projects.stream();
        stream = stream.filter(p -> projectIds.contains(p.getId()));
        if (!abcFilter.isEmpty()) {
            stream = stream.filter(project -> abcFilter.equals(project.getAbcService()));
        }
        if (!text.isEmpty()) {
            stream = filterProjectsByText(text, stream);
        }
        return stream
                .sorted(Comparator.comparing(Project::getId, TextWithNumbersComparator.instance))
                .collect(Collectors.toList());
    }

    private static Stream<Project> filterProjectsByText(String text, Stream<Project> stream) {
        String textLowerCase = text.toLowerCase();
        stream = stream.filter(p -> p.getName().toLowerCase().contains(textLowerCase)
            || p.getId().toLowerCase().contains(textLowerCase));
        return stream;
    }

    private static boolean hasPermissionsInProject(
            Project project,
            String login,
            @Nullable Set<ProjectPermission> permissions)
    {
        if (permissions == null || permissions.isEmpty()) {
            return true;
        }

        if (login.equals(project.getOwner())) {
            return true;
        }

        Acl acl = project.getAcl();

        // We don't support old permissions (UPDATE, DELETE) here
        return permissions.contains(ProjectPermission.READ) && acl.getCanRead().contains(login)
            || permissions.contains(ProjectPermission.CONFIG_UPDATE) && acl.getCanUpdate().contains(login)
            || permissions.contains(ProjectPermission.CONFIG_DELETE) && acl.getCanDelete().contains(login)
            || permissions.contains(ProjectPermission.WRITE) && acl.getCanWrite().contains(login);
    }

    /**
     * PROJECTS TABLE
     */
    private static final class ProjectsTable extends YdbTable<String, Project> {

        private final ObjectMapper objectMapper;

        ProjectsTable(TableClient tableClient, String path, ObjectMapper objectMapper, Executor executor) {
            super(tableClient, path, executor);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("owner", PrimitiveType.utf8())
                .addNullableColumn("abcService", PrimitiveType.utf8())
                .addNullableColumn("readAcl", PrimitiveType.utf8())
                .addNullableColumn("updateAcl", PrimitiveType.utf8())
                .addNullableColumn("deleteAcl", PrimitiveType.utf8())
                .addNullableColumn("writeAcl", PrimitiveType.utf8())
                .addNullableColumn("onlyAuthRead", PrimitiveType.bool())
                .addNullableColumn("onlyAuthPush", PrimitiveType.bool())
                .addNullableColumn("onlySensorNameShards", PrimitiveType.bool())
                .addNullableColumn("onlyNewFormatWrites", PrimitiveType.bool())
                .addNullableColumn("onlyNewFormatReads", PrimitiveType.bool())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("metricNameLabel", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKey("id")
                .build();
        }

        @Override
        protected String getId(Project project) {
            return project.getId();
        }

        @Override
        protected Params toParams(Project project) {
            return Params.create()
                .put("$id", utf8(project.getId()))
                .put("$name", utf8(project.getName()))
                .put("$description", utf8(project.getDescription()))
                .put("$owner", utf8(project.getOwner()))
                .put("$abcService", utf8(project.getAbcService()))
                .put("$readAcl", utf8(toTsv(project.getAcl().getCanRead())))
                .put("$updateAcl", utf8(toTsv(project.getAcl().getCanUpdate())))
                .put("$deleteAcl", utf8(toTsv(project.getAcl().getCanDelete())))
                .put("$writeAcl", utf8(toTsv(project.getAcl().getCanWrite())))
                .put("$onlyAuthRead", bool(project.isOnlyAuthRead()))
                .put("$onlyAuthPush", bool(project.isOnlyAuthPush()))
                .put("$onlySensorNameShards", bool(project.isOnlyMetricNameShards()))
                .put("$onlyNewFormatWrites", bool(project.isOnlyNewFormatWrites()))
                .put("$onlyNewFormatReads", bool(project.isOnlyNewFormatReads()))
                .put("$labels", utf8(Nullables.orEmpty(toJson(objectMapper, project.getLabels()))))
                .put("$metricNameLabel", utf8(project.getMetricNameLabel()))
                .put("$createdAt", int64(project.getCreatedAtMillis()))
                .put("$updatedAt", int64(project.getUpdatedAtMillis()))
                .put("$createdBy", utf8(project.getCreatedBy()))
                .put("$updatedBy", utf8(project.getUpdatedBy()))
                .put("$version", int32(project.getVersion()));
        }

        @Override
        protected Project mapFull(ResultSetReader r) {
            return partialProject(Project.newBuilder(), r)
                .setDescription(r.getColumn("description").getUtf8())
                .setAbcService(r.getColumn("abcService").getUtf8())
                .setAcl(toAcl(r))
                .setOnlyAuthRead(r.getColumn("onlyAuthRead").getBool())
                .setOnlyAuthPush(r.getColumn("onlyAuthPush").getBool())
                .setOnlyMetricNameShards(r.getColumn("onlySensorNameShards").getBool())
                .setOnlyNewFormatWrites(r.getColumn("onlyNewFormatWrites").getBool())
                .setOnlyNewFormatReads(r.getColumn("onlyNewFormatReads").getBool())
                .setLabels(fromJsonMap(objectMapper, r.getColumn("labels").getUtf8()))
                .setMetricNameLabel(r.getColumn("metricNameLabel").getUtf8())
                .setCreatedAtMillis(r.getColumn("createdAt").getInt64())
                .setCreatedBy(r.getColumn("createdBy").getUtf8())
                .setUpdatedAtMillis(r.getColumn("updatedAt").getInt64())
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                .setVersion(r.getColumn("version").getInt32())
                .build();
        }

        @Override
        protected Project mapPartial(ResultSetReader r) {
            return mapFull(r);
        }

        private static Project.Builder partialProject(Project.Builder builder, ResultSetReader r) {
            return builder
                .setId(r.getColumn("id").getUtf8())
                .setName(r.getColumn("name").getUtf8())
                .setOwner(r.getColumn("owner").getUtf8());
        }

        private static Acl toAcl(ResultSetReader r) {
            return Acl.of(
                setFromTsv(r.getColumn("readAcl").getUtf8()),
                setFromTsv(r.getColumn("updateAcl").getUtf8()),
                setFromTsv(r.getColumn("deleteAcl").getUtf8()),
                setFromTsv(r.getColumn("writeAcl").getUtf8()));
        }
    }

    @ParametersAreNonnullByDefault
    public static record PageToken(
            @JsonProperty("size") int size,
            @JsonProperty("current") int current)
    {

        private static final PageTokenCodec<PageToken> codec = PageTokenCodec.forType(PageToken.class);

        public static PageToken decode(CharSequence token, long pageSize) {
            if (pageSize == 0) {
                pageSize = 100;
            }
            return token.isEmpty() ? new PageToken((int) pageSize, 0) : codec.decode(token);
        }

        public String encode() {
            return codec.encode(this);
        }

        public PageOptions toPageOptions() {
            return new PageOptions(size, current);
        }
    }

    @ParametersAreNonnullByDefault
    public static record PageTokenLastId(@JsonProperty("last") String lastId)
    {
        private static final PageTokenCodec<PageTokenLastId> codec = PageTokenCodec.forType(PageTokenLastId.class);

        public static PageTokenLastId decode(CharSequence token) {
            return token.isEmpty() ? new PageTokenLastId("") : codec.decode(token);
        }

        public String encode() {
            return codec.encode(this);
        }
    }
}
