package ru.yandex.solomon.core.db.dao.ydb;

import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.db.dao.QuotasDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.Quota;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class YdbQuotasDao implements QuotasDao {
    private static final OptionalType OPTIONAL_UTF8 = OptionalType.of(PrimitiveType.utf8());
    private static final QueryTemplate TEMPLATE = new QueryTemplate("quota", Arrays.asList(
        "find_by_namespace",
        "find_or_defaults",
        "upsert",
        "delete",
        "delete_one"
    ));

    private final QuotasTable table;
    private final QueryText queryText;

    public YdbQuotasDao(TableClient tableClient, String tablePath) {
        this.table = new QuotasTable(tableClient, tablePath);
        this.queryText = TEMPLATE.build(Map.of("quota.table.path", tablePath));
    }

    @Override
    public CompletableFuture<List<Quota>> findAllByNamespace(String namespace) {
        try {
            String query = queryText.query("find_by_namespace");
            Params params = Params.of("$namespace", utf8(namespace));
            return table.queryList(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Quota>> findAllIndicators(String namespace, String scopeType, @Nullable String scopeId) {
        try {
            String query = queryText.query("find_or_defaults");
            Params params = Params.of(
                "$namespace", utf8(namespace),
                "$scopeType", utf8(scopeType),
                "$scopeId", optionalUtf8(scopeId));
            return table.queryList(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }
    @Override
    public CompletableFuture<Void> upsert(String namespace, String scopeType, @Nullable String scopeId, String indicator, long newLimit, String updatedBy, Instant updatedAt) {
        try {
            String query = queryText.query("upsert");
            Params params = Params.create()
                .put("$namespace", utf8(namespace))
                .put("$scopeType", utf8(scopeType))
                .put("$scopeId", optionalUtf8(scopeId))
                .put("$indicator", utf8(indicator))
                .put("$limit", int64(newLimit))
                .put("$updatedBy", utf8(updatedBy))
                .put("$updatedAt", int64(updatedAt.toEpochMilli()));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteOne(String namespace, String scopeType, @Nullable String scopeId, String indicator) {
        try {
            String query = queryText.query("delete_one");
            Params params = Params.create()
                .put("$namespace", utf8(namespace))
                .put("$scopeType", utf8(scopeType))
                .put("$scopeId", optionalUtf8(scopeId))
                .put("$indicator", utf8(indicator));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> delete(String namespace, String scopeType, String indicator) {
        try {
            String query = queryText.query("delete");
            Params params = Params.create()
                .put("$namespace", utf8(namespace))
                .put("$scopeType", utf8(scopeType))
                .put("$indicator", utf8(indicator));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static final class QuotasTable extends YdbTable<String, Quota> {

        QuotasTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("namespace", PrimitiveType.utf8())
                .addNullableColumn("scopeType", PrimitiveType.utf8())
                .addNullableColumn("scopeId", PrimitiveType.utf8())
                .addNullableColumn("indicator", PrimitiveType.utf8())
                .addNullableColumn("limit", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .setPrimaryKeys("namespace", "scopeType", "scopeId", "indicator")
                .build();
        }

        @Override
        protected String getId(Quota quota) {
            if (quota.isDefaultScopeId()) {
                return quota.getNamespace() + "_" + quota.getScopeType() + "_" + quota.getIndicator();
            }
            return quota.getNamespace() + "_" + quota.getScopeType() + "_" + quota.getScopeId() + "_" + quota.getIndicator();
        }

        @Override
        protected Params toParams(Quota quota) {
            return Params.create()
                .put("$namespace", utf8(quota.getNamespace()))
                .put("$scopeType", utf8(quota.getScopeType()))
                .put("$scopeId", optionalUtf8(quota.getScopeId()))
                .put("$indicator", utf8(quota.getIndicator()))
                .put("$limit", int64(quota.getLimit()))
                .put("$updatedAt", int64(quota.getUpdatedAt().toEpochMilli()))
                .put("$updatedBy", utf8(quota.getUpdatedBy()));
        }

        @Override
        protected Quota mapFull(ResultSetReader r) {
            return quota(Quota.newBuilder(), r).build();
        }

        @Override
        protected Quota mapPartial(ResultSetReader r) {
            return mapFull(r);
        }

        private static Quota.Builder quota(Quota.Builder builder, ResultSetReader r) {
            var scopeId = r.getColumn("scopeId");
            if (scopeId.isOptionalItemPresent()) {
                builder.setScopeId(scopeId.getUtf8());
            }
            return builder
                .setNamespace(r.getColumn("namespace").getUtf8())
                .setScopeType(r.getColumn("scopeType").getUtf8())
                .setIndicator(r.getColumn("indicator").getUtf8())
                .setLimit(r.getColumn("limit").getInt64())
                .setUpdatedAt(Instant.ofEpochMilli(r.getColumn("updatedAt").getInt64()))
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8());
        }
    }

    private static OptionalValue optionalUtf8(@Nullable String value) {
        if (value == null) {
            return OPTIONAL_UTF8.emptyValue();
        }
        return OPTIONAL_UTF8.newValue(utf8(value));
    }
}
