package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AlterTableSettings;
import com.yandex.ydb.table.values.PrimitiveType;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.ReferenceConf;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonList;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.listFromTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsv;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class YdbServiceProvidersDao implements ServiceProvidersDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbServiceProvidersDao.class);

    private static final QueryTemplate TEMPLATE = new QueryTemplate("service_provider", Arrays.asList(
            "read",
            "exists",
            "list",
            "insert_with_globalId",
            "update_partial_with_globalId",
            "delete"
    ));

    private final ServiceProvidersTable table;
    private final QueryText queryText;

    public YdbServiceProvidersDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new ServiceProvidersTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("service.provider.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Boolean> insert(ServiceProvider serviceProvider) {
        try {
            String query = queryText.query("insert_with_globalId");
            return table.insertOne(query, serviceProvider);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<ServiceProvider>> read(String id) {
        try {
            String query = queryText.query("read");
            Params params = Params.of("$id", utf8(id));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<ServiceProvider>> update(ServiceProvider serviceProvider) {
        try {
            String query = queryText.query("update_partial_with_globalId");
            return table.updateOne(query, serviceProvider);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(String id) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$id", utf8(id));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<ServiceProvider>> list(String filterById, int pageSize, String pageToken) {
        try {
            String query = queryText.query("list");

            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$filterRegexp", utf8(toRegularExpression(filterById)),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(query, params)
                    .thenApply(result -> {
                        final String nextPageToken;

                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }

                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String id) {
        try {
            String query = queryText.query("exists");
            Params params = Params.of("$id", utf8(id));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<ServiceProvider>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    public CompletableFuture<Void> migrateSchema() {
        return table.retryCtx().supplyResult(session -> session.describeTable(table.getPath()))
                .thenCompose(result -> {
                    if (!result.isSuccess()) {
                        return createSchemaForTests();
                    }
                    return alterTable(result.expect("unable describe " + table.getPath()));
                });
    }

    private CompletableFuture<Void> alterTable(TableDescription tableDescription) {
        for (var column : tableDescription.getColumns()) {
            if ("hasGlobalId".equals(column.getName())) {
                return completedFuture(null);
            }
        }

        var settings = new AlterTableSettings();
        settings.addColumn("hasGlobalId", PrimitiveType.bool().makeOptional());
        return table.retryCtx().supplyStatus(session -> session.alterTable(table.getPath(), settings))
                .thenAccept(status -> status.expect("unable alter " + table.getPath()));
    }

    private static class ServiceProvidersTable extends YdbTable<String, ServiceProvider> {
        private final ObjectMapper objectMapper;
        protected ServiceProvidersTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .addNullableColumn("description", PrimitiveType.utf8())
                    .addNullableColumn("shardSettings", PrimitiveType.utf8())
                    .addNullableColumn("referenceSettings", PrimitiveType.utf8())
                    .addNullableColumn("abcService", PrimitiveType.utf8())
                    .addNullableColumn("cloudId", PrimitiveType.utf8())
                    .addNullableColumn("tvmDestId", PrimitiveType.utf8())
                    .addNullableColumn("iamServiceAccountId", PrimitiveType.utf8())
                    .addNullableColumn("tvmServiceIds", PrimitiveType.utf8())
                    .addNullableColumn("createdAt", PrimitiveType.int64())
                    .addNullableColumn("updatedAt", PrimitiveType.int64())
                    .addNullableColumn("createdBy", PrimitiveType.utf8())
                    .addNullableColumn("updatedBy", PrimitiveType.utf8())
                    .addNullableColumn("version", PrimitiveType.int32())
                    .addNullableColumn("hasGlobalId", PrimitiveType.bool())
                    .setPrimaryKey("id")
                    .build();
        }

        @Override
        protected String getId(ServiceProvider serviceProvider) {
            return serviceProvider.getId();
        }

        @Override
        protected Params toParams(ServiceProvider serviceProvider) {
            var tvmIdStrings = serviceProvider.getTvmServiceIds().stream().map(e -> e.toString()).collect(Collectors.toList());

            return Params.create()
                    .put("$id", utf8(serviceProvider.getId()))
                    .put("$description", utf8(serviceProvider.getDescription()))
                    .put("$shardSettings", utf8(toJson(objectMapper, serviceProvider.getShardSettings())))
                    .put("$referenceSettings", utf8(toJson(objectMapper, serviceProvider.getReferences())))
                    .put("$abcService", utf8(serviceProvider.getAbcService()))
                    .put("$cloudId", utf8(serviceProvider.getCloudId()))
                    .put("$tvmDestId", utf8(serviceProvider.getTvmDestId()))
                    .put("$iamServiceAccountId", utf8(toTsv(serviceProvider.getIamServiceAccountIds())))
                    .put("$tvmServiceIds", utf8(toTsv(tvmIdStrings)))
                    .put("$createdAt", int64(serviceProvider.getCreatedAtMillis()))
                    .put("$updatedAt", int64(serviceProvider.getUpdatedAtMillis()))
                    .put("$createdBy", utf8(serviceProvider.getCreatedBy()))
                    .put("$updatedBy", utf8(serviceProvider.getUpdatedBy()))
                    .put("$hasGlobalId", bool(serviceProvider.isHasGlobalId()))
                    .put("$version", int32(serviceProvider.getVersion()));
        }

        @Override
        protected ServiceProvider mapFull(ResultSetReader resultSet) {
            return ServiceProvider.newBuilder()
                    .setId(resultSet.getColumn("id").getUtf8())
                    .setDescription(resultSet.getColumn("description").getUtf8())
                    .setShardSettings(read(resultSet, "shardSettings", ServiceProviderShardSettings.class))
                    .setReferences(readList(resultSet, "referenceSettings", ReferenceConf.class))
                    .setAbcService(resultSet.getColumn("abcService").getUtf8())
                    .setCloudId(resultSet.getColumn("cloudId").getUtf8())
                    .setTvmDestId(resultSet.getColumn("tvmDestId").getUtf8())
                    .setIamServiceAccountIds(listFromTsv(resultSet.getColumn("iamServiceAccountId").getUtf8()))
                    .setTvmServiceIds(new IntArrayList(
                            listFromTsv(resultSet.getColumn("tvmServiceIds").getUtf8())
                                    .stream().map(Integer::valueOf).iterator()))
                    .setCreatedAtMillis(resultSet.getColumn("createdAt").getInt64())
                    .setUpdatedAtMillis(resultSet.getColumn("updatedAt").getInt64())
                    .setCreatedBy(resultSet.getColumn("createdBy").getUtf8())
                    .setUpdatedBy(resultSet.getColumn("updatedBy").getUtf8())
                    .setVersion(resultSet.getColumn("version").getInt32())
                    .setHasGlobalId(resultSet.getColumn("hasGlobalId").getBool())
                    .build();
        }

        private <T> T read(ResultSetReader resultSet, String column, Class<T> clazz) {
            String json = resultSet.getColumn(column).getUtf8();
            try {
                return fromJson(objectMapper, json, clazz);
            } catch (Throwable e) {
                logger.error("failed to parse at column {} json {} in service provider", column, json);
                return null;
            }
        }

        private <T> List<T> readList(ResultSetReader resultSet, String columnName, Class<T> clazz) {
            String json = resultSet.getColumn(columnName).getUtf8();
            return fromJsonList(objectMapper, json, clazz);
        }

        @Override
        protected ServiceProvider mapPartial(ResultSetReader resultSet) {
            return mapFull(resultSet);
        }
    }
}
