package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonMap;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonOrNull;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;


/**
 * @author Sergey Polovko
 */
public class YdbServicesDao implements ServicesDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbServicesDao.class);
    private static final QueryTemplate TEMPLATE = new QueryTemplate("service", Arrays.asList(
        "insert",
        "find_with_project",
        "paged_find",
        "paged_find_v3",
        "paged_find_all",
        "paged_find_all_lite",
        "update_partial",
        "delete_with_project",
        "delete_for_project",
        "exists_with_project"
    ));

    private final ServicesTable table;
    private final QueryText queryText;

    public YdbServicesDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper, Executor executor) {
        this.table = new ServicesTable(tableClient, tablePath, objectMapper, executor);
        this.queryText = TEMPLATE.build(Collections.singletonMap("service.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Boolean> insert(Service service) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, service);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Service>> findOne(String projectId, String folderId, String serviceId) {
        try {
            String query = queryText.query("find_with_project");
            Params params = Params.of("$id", utf8(serviceId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<PagedResult<Service>> findByProjectId(
            String projectId, String folderId, PageOptions pageOpts, String text, ShardSettings.Type monitoringModel)
    {
        String modelFilter = monitoringModel == ShardSettings.Type.UNSPECIFIED
                ? ""
                : "\"type\":\"" + monitoringModel.name() + "\"";
        try {
            Params params = Params.of("$projectId", utf8(projectId),
                    "$folderId", utf8(folderId),
                    "$regexp", utf8(toRegularExpression(text)),
                    "$modelFilter", utf8(modelFilter));
            return table.queryPage(params, pageOpts, opts -> {
                return queryText.query("paged_find", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Service>> findByProjectIdPaged(
            String projectId,
            String folderId,
            int pageSize, String pageToken,
            String text) {
        try {
            TokenPageOptions pageOpts = new TokenPageOptions(pageSize, pageToken);
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId), "$regexp", utf8(toRegularExpression(text)));
            return table.queryPage(params, pageOpts, opts -> queryText.query("paged_find_v3", paging(opts)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Service>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<PagedResult<Service>> findAll(PageOptions pageOpts, String text) {
        boolean useRegex = StringUtils.isNoneBlank(text);
        if (!pageOpts.isLimited() && !useRegex) {
            return table.queryAll().thenApply(result -> PagedResult.of(result, pageOpts, result.size()));
        }

        try {
            Params params = Params.create();

            if (useRegex) {
                params.put("$regexp", utf8(toRegularExpression(text)));
            }
            return table.queryPage(params, pageOpts, opts ->
                queryText.query(useRegex ? "paged_find_all" : "paged_find_all_lite", paging(opts))
            );
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Service>> partialUpdate(Service service) {
        try {
            String query = queryText.query("update_partial");
            return table.updateOne(query, service);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String serviceId) {
        try {
            String query = queryText.query("delete_with_project");
            Params params = Params.of("$id", utf8(serviceId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteByProjectId(String projectId, String folderId) {
        try {
            String query = queryText.query("delete_for_project");
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String projectId, String folderId, String serviceId) {
        try {
            String query = queryText.query("exists_with_project");
            Params params = Params.of("$id", utf8(serviceId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * SERVICES TABLE
     */
    private static final class ServicesTable extends YdbTable<String, Service> {

        private final ObjectMapper objectMapper;

        ServicesTable(TableClient tableClient, String path, ObjectMapper objectMapper, Executor executor) {
            super(tableClient, path, executor);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("serviceProvider", PrimitiveType.utf8())
                .addNullableColumn("type", PrimitiveType.utf8()) // TODO: drop column
                .addNullableColumn("port", PrimitiveType.int32())
                .addNullableColumn("path", PrimitiveType.utf8())
                .addNullableColumn("addTsArgs", PrimitiveType.bool())
                .addNullableColumn("interval", PrimitiveType.int32())
                .addNullableColumn("gridSec", PrimitiveType.int32())
                .addNullableColumn("protobufType", PrimitiveType.utf8()) // TODO: drop column
                .addNullableColumn("sensorConf", PrimitiveType.utf8())
                .addNullableColumn("sensorsTtlDays", PrimitiveType.int32())
                .addNullableColumn("sensorNameLabel", PrimitiveType.utf8())
                .addNullableColumn("tvmDestId", PrimitiveType.utf8())
                .addNullableColumn("shardSettings", PrimitiveType.utf8())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected String getId(Service service) {
            return service.getId();
        }

        @Override
        protected Params toParams(Service service) {
            final ShardSettings settings = service.getShardSettings();
            final int port = ShardSettings.getPort(settings, 0);
            final String path = ShardSettings.getPath(settings, "");
            final int interval = ShardSettings.getInterval(settings, Service.DEFAULT_INTERVAL);
            final boolean addTsArgs = settings.getPullSettings() != null && settings.getPullSettings().isAddTsArgs();
            final String tvmDestId = settings.getPullSettings() != null ? settings.getPullSettings().getTvmDestinationId() : "";
            final int grid = settings.getGrid();
            final int ttl = settings.getMetricsTtl();
            final ServiceMetricConf conf = ShardSettings.toServiceMetricConf(settings, ServiceMetricConf.empty());

            return Params.create()
                .put("$id", utf8(service.getId()))
                .put("$projectId", utf8(service.getProjectId()))
                .put("$folderId", utf8(service.getFolderId()))
                .put("$name", utf8(service.getName()))
                .put("$description", utf8(service.getDescription()))
                .put("$serviceProvider", utf8(service.getServiceProvider()))
                .put("$port", int32(port))
                .put("$path", utf8(path))
                .put("$addTsArgs", bool(addTsArgs))
                .put("$interval", int32(interval))
                .put("$gridSec", int32(grid))
                .put("$sensorConf", utf8(toJson(objectMapper, conf)))
                .put("$sensorsTtlDays", int32(ttl))
                .put("$sensorNameLabel", utf8(service.getMetricNameLabel()))
                .put("$tvmDestId", utf8(tvmDestId))
                .put("$shardSettings", utf8(Nullables.orEmpty(toJson(objectMapper, settings))))
                .put("$labels", utf8(Nullables.orEmpty(toJson(objectMapper, service.getLabels()))))
                .put("$createdAt", int64(service.getCreatedAtMillis()))
                .put("$updatedAt", int64(service.getUpdatedAtMillis()))
                .put("$createdBy", utf8(service.getCreatedBy()))
                .put("$updatedBy", utf8(service.getUpdatedBy()))
                .put("$version", int32(service.getVersion()));
        }

        @Override
        protected Service mapFull(ResultSetReader r) {
            return partialService(Service.newBuilder(), r)
                .setDescription(r.getColumn("description").getUtf8())
                .setServiceProvider(r.getColumn("serviceProvider").getUtf8())
                .setAddTsArgs(r.getColumn("addTsArgs").getBool())
                .setMetricNameLabel(r.getColumn("sensorNameLabel").getUtf8())
                .setLabels(fromJsonMap(objectMapper, r.getColumn("labels").getUtf8()))
                .setCreatedBy(r.getColumn("createdBy").getUtf8())
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                .setVersion(r.getColumn("version").getInt32())
                .build();
        }

        @Override
        protected Service mapPartial(ResultSetReader resultSet) {
            return partialService(Service.newBuilder(), resultSet).build();
        }

        private Service.Builder partialService(Service.Builder builder, ResultSetReader resultSet) {
            ServiceMetricConf metricConf;
            try {
                metricConf = fromJson(objectMapper, resultSet.getColumn("sensorConf").getUtf8(), ServiceMetricConf.class);
            } catch (Throwable e) {
                logger.error("At projectId {} for serviceId {} failed to parse metric conf {}",
                        resultSet.getColumn("projectId").getUtf8(),
                        resultSet.getColumn("id").getUtf8(),
                        resultSet.getColumn("sensorConf").getUtf8(),
                        e);
                metricConf = ServiceMetricConf.empty();
            }

            var serviceBuilder =  builder
                .setId(resultSet.getColumn("id").getUtf8())
                .setProjectId(resultSet.getColumn("projectId").getUtf8())
                .setFolderId(resultSet.getColumn("folderId").getUtf8())
                .setName(resultSet.getColumn("name").getUtf8())
                .setMetricConf(metricConf)
                .setCreatedAtMillis(resultSet.getColumn("createdAt").getInt64())
                .setUpdatedAtMillis(resultSet.getColumn("updatedAt").getInt64());
            return serviceBuilder.setShardSettings(getShardSettings(resultSet, metricConf));
        }

        private ShardSettings getShardSettings(ResultSetReader r, ServiceMetricConf metricConf) {
            var shardSettingsPersisted = fromJsonOrNull(objectMapper, r.getColumn("shardSettings").getUtf8(), ShardSettings.class);
            int port = ShardSettings.getPort(shardSettingsPersisted, 0);
            String path = ShardSettings.getPath(shardSettingsPersisted, "");
            int interval = ShardSettings.getInterval(shardSettingsPersisted, 0);
            int ttl = ShardSettings.getMetricsTtlDays(shardSettingsPersisted, 0);
            boolean addTsArgs = ShardSettings.getAddTsArgs(shardSettingsPersisted, false);
            String tvmDestId = ShardSettings.getTvmDestinationId(shardSettingsPersisted, "");
            int gridSec = ShardSettings.getGrid(shardSettingsPersisted, Service.GRID_UNKNOWN);
            final ShardSettings.Type type;
            final ShardSettings.PullSettings pullSettings;
            if (path.isEmpty()) {
                type = ShardSettings.Type.PUSH;
                pullSettings = null;
            } else {
                type = ShardSettings.Type.PULL;
                pullSettings = ShardSettings.PullSettings.newBuilder()
                        .setPath(path)
                        .setPort(port)
                        .setAddTsArgs(addTsArgs)
                        .setProtocol(ShardSettings.getProtocol(shardSettingsPersisted, ShardSettings.PullProtocol.UNKNOWN))
                        .setTvmDestinationId(tvmDestId)
                        .build();
            }
            return ShardSettings.of(
                    type,
                    pullSettings,
                    gridSec,
                    ttl,
                    ShardSettings.getDecimPolicy(shardSettingsPersisted, DecimPolicy.UNDEFINED),
                    ShardSettings.AggregationSettings.of(true, metricConf.getAggrRules(), metricConf.isRawDataMemOnly()),
                    interval
            );
        }
    }
}
