package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.PrimitiveType;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonMap;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonOrNull;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;


/**
 * @author Sergey Polovko
 */
public class YdbShardsDao implements ShardsDao {

    private static final String PCS_KEY_TABLE_NAME_SUFFIX = "PcsKey";
    private static final String PCS_NUM_ID_TABLE_NAME_SUFFIX = "PcsNumId";

    private static final QueryTemplate TEMPLATE = new QueryTemplate("shard", Arrays.asList(
        "insert",
        "find_with_project",
        "find_by_key",
        "paged_find",
        "paged_find_v3",
        "paged_find_lite",
        "paged_find_all",
        "paged_find_all_lite",
        "update_partial",
        "patch_cluster_name",
        "patch_service_name",
        "delete_with_project",
        "find_by_project",
        "find_by_cluster",
        "find_by_cluster_v3",
        "find_by_service",
        "find_by_service_v3",
        "exists_with_project",
        "release_num_id"
    ));

    private final TableClient tableClient;
    private final ShardsTable table;
    private final QueryText queryText;

    public YdbShardsDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper, Executor executor) {
        this.tableClient = tableClient;
        this.table = new ShardsTable(tableClient, tablePath, objectMapper, executor);
        this.queryText = TEMPLATE.build(Map.of(
            "shard.table.path", tablePath,
            "shard.pcs.key.table.path", table.getPcsIndexPath(),
            "shard.pcs.numId.table.path", table.getNumIdIndexPath()));
    }

    @Override
    public CompletableFuture<Boolean> insert(Shard shard) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, shard);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Shard>> findOne(String projectId, String folderId, String shardId) {
        try {
            String query = queryText.query("find_with_project");
            Params params = Params.of("$id", utf8(shardId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Shard>> findByShardKey(String projectId, String clusterName, String serviceName) {
        try {
            String query = queryText.query("find_by_key");
            Params params = Params.of(
                "$projectId", utf8(projectId),
                "$clusterName", utf8(clusterName),
                "$serviceName", utf8(serviceName));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<PagedResult<Shard>> findByProjectId(
            String projectId, String folderId, PageOptions pageOpts, EnumSet<ShardState> state, String text)
    {
        try {
            Params params = Params.create()
                .put("$projectId", utf8(projectId))
                .put("$folderId", utf8(folderId));

            String filter = state == null || state.isEmpty()
                    ? "true"
                    : state.stream()
                    .map(shardState -> "state = '" + shardState.getName() + "' OR state = '" + shardState.getAnotherName() + "'")
                    .collect(Collectors.joining(" OR "));

            boolean useRegex = StringUtils.isNoneBlank(text);
            if (useRegex) {
                params.put("$regexp", utf8(toRegularExpression(text)));
            }
            return table.queryPage(params, pageOpts, (opts) -> {
                    var map = new HashMap<>(paging(opts));
                    map.put("stateFilter", filter);
                    return queryText.query(useRegex ? "paged_find" : "paged_find_lite", map);
                });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Shard>> findByProjectId(String projectId, String folderId) {
        try {
            String query = queryText.query("find_by_project");
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryList(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Shard>> findByProjectIdV3(
            String projectId,
            String folderId,
            int pageSize,
            String pageToken,
            String text)
    {
        try {
            TokenPageOptions pageOpts = new TokenPageOptions(pageSize, pageToken);
            Params params = Params.of(
                    "$projectId", utf8(projectId),
                    "$folderId", utf8(folderId),
                    "$regexp", utf8(toRegularExpression(text)));
            return table.queryPage(params, pageOpts, opts -> queryText.query("paged_find_v3", paging(opts)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Shard>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<PagedResult<Shard>> findAll(PageOptions pageOpts, ShardState state, String text) {
        boolean useRegex = StringUtils.isNoneBlank(text);
        if (!pageOpts.isLimited() && !useRegex) {
            return table.queryAll(r -> parseState(r.getColumn("state").getUtf8()) == state)
                .thenApply(result -> PagedResult.of(result, pageOpts, result.size()));
        }

        try {
            Params params = Params.create()
                .put("$state", utf8(state.getName()))
                .put("$oldState", utf8(state.getAnotherName()));

            if (useRegex) {
                params.put("$regexp", utf8(toRegularExpression(text)));
            }
            return table.queryPage(params, pageOpts, (opts) -> {
                return queryText.query(useRegex ? "paged_find_all" : "paged_find_all_lite", paging(opts));
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Shard>> findAllNotInactive() {
        return table.queryAll(r -> {
            ShardState state = parseState(r.getColumn("state").getUtf8());
            return state != ShardState.INACTIVE;
        });
    }

    @Override
    public CompletableFuture<Optional<Shard>> partialUpdate(Shard shard, boolean canUpdateInternals) {
        try {
            final ShardSettings settings = shard.getShardSettings();
            final DecimPolicy decimPolicy = ShardSettings.getDecimPolicy(settings, DecimPolicy.UNDEFINED);
            final int ttl = settings.getMetricsTtl();

            Params params = Params.create()
                .put("$id", utf8(shard.getId()))
                .put("$projectId", utf8(shard.getProjectId()))
                .put("$folderId", utf8(shard.getFolderId()))
                .put("$sensorsTtlDays", int32(ttl))
                .put("$sensorNameLabel", utf8(shard.getMetricNameLabel()).makeOptional())
                .put("$description", utf8(shard.getDescription()))
                .put("$shardSettings", utf8(Nullables.orEmpty(toJson(table.objectMapper, settings))))
                .put("$labels", utf8(Nullables.orEmpty(toJson(table.objectMapper, shard.getLabels()))))
                .put("$state", utf8(shard.getState().getName()).makeOptional())
                .put("$updatedAt", int64(shard.getUpdatedAtMillis()))
                .put("$updatedBy", utf8(shard.getUpdatedBy()))
                .put("$version", int32(shard.getVersion()));

            if (canUpdateInternals) {
                params.put("$maxSensorsPerUrl", int32(shard.getMaxMetricsPerUrl()).makeOptional());
                params.put("$maxFileSensors", int32(shard.getMaxFileMetrics()).makeOptional());
                params.put("$maxMemSensors", int32(shard.getMaxMemMetrics()).makeOptional());
                params.put("$maxResponseSizeBytes", int32(shard.getMaxResponseSizeBytes()).makeOptional());
                params.put("$numPartitions", int32(shard.getNumPartitions()).makeOptional());
                params.put("$decimPolicy", utf8(decimPolicy.name()).makeOptional());
                params.put("$validationMode", utf8(shard.getValidationMode().getName()).makeOptional());
            }
            String query = queryText.query("update_partial");
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> patchWithClusterName(String projectId, String clusterId, String clusterName) {
        try {
            String query = queryText.query("patch_cluster_name");
            Params params = Params.of(
                "$projectId", utf8(projectId),
                "$clusterId", utf8(clusterId),
                "$clusterName", utf8(clusterName));

            return table.execute(query, params)
                .thenAccept(r -> {
                    if (!r.isSuccess() && r.getCode() == StatusCode.PRECONDITION_FAILED) {
                        throw new ConflictException("failed to update cluster name because shard with same cluster and service names already exists");
                    }
                });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> patchWithServiceName(String projectId, String serviceId, String serviceName) {
        try {
            String query = queryText.query("patch_service_name");
            Params params = Params.of(
                "$projectId", utf8(projectId),
                "$serviceId", utf8(serviceId),
                "$serviceName", utf8(serviceName));

            return table.execute(query, params)
                .thenAccept(r -> {
                    if (!r.isSuccess() && r.getCode() == StatusCode.PRECONDITION_FAILED) {
                        throw new ConflictException("failed to update service name because shard with same cluster and service names already exists");
                    }
                });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String projectId, String folderId, String shardId) {
        try {
            String query = queryText.query("delete_with_project");
            Params params = Params.of("$id", utf8(shardId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> releaseNumId(String projectId, String shardId, int numId) {
        return CompletableFutures.safeCall(() -> {
            String query = queryText.query("release_num_id");
            Params params = Params.of("$id", utf8(shardId), "$projectId", utf8(projectId), "$numId", int32(numId));
            return table.queryBool(query, params);
        });
    }

    @Override
    public CompletableFuture<List<Shard>> findByClusterId(String projectId, String folderId, String clusterId) {
        try {
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId), "$clusterId", utf8(clusterId));
            return table.queryPage(params, PageOptions.ALL, (pageOpts) -> {
                    return queryText.query("find_by_cluster", paging(pageOpts));
                })
                .thenApply(PagedResult::getResult);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Shard>> findByClusterIdV3(
            String projectId,
            String folderId,
            String clusterId,
            Set<ShardState> states,
            String filter,
            int pageSize,
            String pageToken) {
        try {
            TokenPageOptions pageOptions = new TokenPageOptions(pageSize, pageToken);

            Params params = Params.of(
                    "$projectId", utf8(projectId),
                    "$folderId", utf8(folderId),
                    "$clusterId", utf8(clusterId),
                    "$filter", utf8(toRegularExpression(filter)),
                    "$states", utf8(toRegularStatesExpression(states)));

            return table.queryPage(params, pageOptions, (pageOpts) ->
                    queryText.query("find_by_cluster_v3", paging(pageOpts)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Shard>> findByServiceId(String projectId, String folderId, String serviceId) {
        try {
            Params params = Params.of("$projectId", utf8(projectId), "$folderId", utf8(folderId), "$serviceId", utf8(serviceId));
            return table.queryPage(params, PageOptions.ALL, (pageOpts) -> queryText.query("find_by_service", paging(pageOpts)))
                .thenApply(PagedResult::getResult);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Shard>> findByServiceIdV3(
            String projectId,
            String folderId,
            String serviceId,
            Set<ShardState> states,
            String filter,
            int pageSize,
            String pageToken) {
        try {
            TokenPageOptions pageOptions = new TokenPageOptions(pageSize, pageToken);

            Params params = Params.of(
                    "$projectId", utf8(projectId),
                    "$folderId", utf8(folderId),
                    "$serviceId", utf8(serviceId),
                    "$filter", utf8(toRegularExpression(filter)),
                    "$states", utf8(toRegularStatesExpression(states)));

            return table.queryPage(params, pageOptions, (pageOpts) ->
                    queryText.query("find_by_service_v3", paging(pageOpts)));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String projectId, String folderId, String shardId) {
        try {
            String query = queryText.query("exists_with_project");
            Params params = Params.of("$id", utf8(shardId), "$projectId", utf8(projectId), "$folderId", utf8(folderId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Int2ObjectMap<String>> findAllIdToShardId() {
        SessionRetryContext retryCtx = SessionRetryContext.create(tableClient)
            .maxRetries(3)
            .build();

        ReadTableSettings settings = ReadTableSettings.newBuilder()
            .timeout(1, TimeUnit.MINUTES)
            .orderedRead(false)
            .build();

        return retryCtx.supplyResult(session -> {
            Int2ObjectMap<String> map = new Int2ObjectOpenHashMap<>();
            return session.readTable(table.getNumIdIndexPath(), settings, (resultSet) -> {
                final int numIdIdx = resultSet.getColumnIndex("numId");
                final int idIdx = resultSet.getColumnIndex("id");
                while (resultSet.next()) {
                    map.put(
                        resultSet.getColumn(numIdIdx).getInt32(),
                        resultSet.getColumn(idIdx).getUtf8());
                }
            })
            .thenApply(s -> s.isSuccess() ? Result.success(map) : Result.fail(s));
        })
        .thenApply(r -> r.expect("cannot read table " + table.getNumIdIndexPath()));
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static ShardState parseState(String stateString) {
        if (ShardState.isNewItemState(stateString)) {
            return ShardState.mapFromString(stateString);
        }
        return ShardState.mapFromOldFormat(ShardState.valueOf(stateString));
    }

    private static String toRegularStatesExpression(Set<ShardState> states) {
        if (states.isEmpty()) {
            return ".*";
        }
        return "(" + states.stream().map(ShardState::getName).collect(Collectors.joining("|")) + ")";
    }

    /**
     * SHARDS TABLE
     */
    private static final class ShardsTable extends YdbTable<Integer, Shard> {
        private final String pcsIndexPath;
        private final String numIdIndexPath;
        private final ObjectMapper objectMapper;

        ShardsTable(TableClient tableClient, String path, ObjectMapper objectMapper, Executor executor) {
            super(tableClient, path, executor);
            this.pcsIndexPath = path + PCS_KEY_TABLE_NAME_SUFFIX;
            this.numIdIndexPath = path + PCS_NUM_ID_TABLE_NAME_SUFFIX;
            this.objectMapper = objectMapper;
        }

        String getPcsIndexPath() {
            return pcsIndexPath;
        }

        String getNumIdIndexPath() {
            return numIdIndexPath;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("numId", PrimitiveType.int32())
                .addNullableColumn("clusterId", PrimitiveType.utf8())
                .addNullableColumn("serviceId", PrimitiveType.utf8())
                .addNullableColumn("clusterName", PrimitiveType.utf8())
                .addNullableColumn("serviceName", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("maxSensorsPerUrl", PrimitiveType.int32())
                .addNullableColumn("maxFileSensors", PrimitiveType.int32())
                .addNullableColumn("maxMemSensors", PrimitiveType.int32())
                .addNullableColumn("maxResponseSizeBytes", PrimitiveType.int32())
                .addNullableColumn("numPartitions", PrimitiveType.int32())
                .addNullableColumn("decimPolicy", PrimitiveType.utf8())
                .addNullableColumn("state", PrimitiveType.utf8())
                .addNullableColumn("validationMode", PrimitiveType.utf8())
                .addNullableColumn("sensorsTtlDays", PrimitiveType.int32())
                .addNullableColumn("sensorNameLabel", PrimitiveType.utf8())
                .addNullableColumn("shardSettings", PrimitiveType.utf8())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected Map<String, TableDescription> indexes() {
            var pcsIndex = TableDescription.newBuilder()
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("clusterName", PrimitiveType.utf8())
                .addNullableColumn("serviceName", PrimitiveType.utf8())
                .addNullableColumn("id", PrimitiveType.utf8())
                .setPrimaryKeys("projectId", "clusterName", "serviceName")
                .build();

            var numIdIndex = TableDescription.newBuilder()
                .addNullableColumn("numId", PrimitiveType.int32())
                .addNullableColumn("id", PrimitiveType.utf8())
                .setPrimaryKey("numId")
                .build();

            return Map.of(pcsIndexPath, pcsIndex, numIdIndexPath, numIdIndex);
        }

        @Override
        protected Integer getId(Shard shard) {
            return shard.getNumId();
        }

        @Override
        protected Params toParams(Shard shard) {
            final ShardSettings settings = shard.getShardSettings();
            final DecimPolicy decimPolicy = ShardSettings.getDecimPolicy(settings, DecimPolicy.UNDEFINED);
            final int ttl = settings.getMetricsTtl();

            return Params.create()
                .put("$id", utf8(shard.getId()))
                .put("$projectId", utf8(shard.getProjectId()))
                .put("$folderId", utf8(shard.getFolderId()))
                .put("$numId", int32(shard.getNumId()))
                .put("$clusterId", utf8(shard.getClusterId()))
                .put("$serviceId", utf8(shard.getServiceId()))
                .put("$clusterName", utf8(shard.getClusterName()))
                .put("$serviceName", utf8(shard.getServiceName()))
                .put("$description", utf8(shard.getDescription()))
                .put("$maxSensorsPerUrl", int32(shard.getMaxMetricsPerUrl()))
                .put("$maxFileSensors", int32(shard.getMaxFileMetrics()))
                .put("$maxMemSensors", int32(shard.getMaxMemMetrics()))
                .put("$maxResponseSizeBytes", int32(shard.getMaxResponseSizeBytes()))
                .put("$numPartitions", int32(shard.getNumPartitions()))
                .put("$decimPolicy", utf8(decimPolicy.name()))
                .put("$state", utf8(shard.getState().getName()))
                .put("$validationMode", utf8(shard.getValidationMode().getName()))
                .put("$sensorsTtlDays", int32(ttl))
                .put("$sensorNameLabel", utf8(shard.getMetricNameLabel()))
                .put("$shardSettings", utf8(Nullables.orEmpty(toJson(objectMapper, settings))))
                .put("$labels", utf8(Nullables.orEmpty(toJson(objectMapper, shard.getLabels()))))
                .put("$createdAt", int64(shard.getCreatedAtMillis()))
                .put("$updatedAt", int64(shard.getUpdatedAtMillis()))
                .put("$createdBy", utf8(shard.getCreatedBy()))
                .put("$updatedBy", utf8(shard.getUpdatedBy()))
                .put("$version", int32(shard.getVersion()));
        }

        @Override
        protected Shard mapFull(ResultSetReader r) {
            var shardBuilder = partialShard(Shard.newBuilder(), r)
                .setDescription(r.getColumn("description").getUtf8())
                .setMaxMetricsPerUrl(r.getColumn("maxSensorsPerUrl").getInt32())
                .setMaxFileMetrics(r.getColumn("maxFileSensors").getInt32())
                .setMaxMemMetrics(r.getColumn("maxMemSensors").getInt32())
                .setMaxResponseSizeBytes(r.getColumn("maxResponseSizeBytes").getInt32())
                .setNumPartitions(r.getColumn("numPartitions").getInt32())
                .setValidationMode(ValidationMode.fromString(r.getColumn("validationMode").getUtf8()))
                .setMetricNameLabel(r.getColumn("sensorNameLabel").getUtf8())
                .setLabels(fromJsonMap(objectMapper, r.getColumn("labels").getUtf8()))
                .setCreatedBy(r.getColumn("createdBy").getUtf8())
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                .setVersion(r.getColumn("version").getInt32());

            return shardBuilder.setShardSettings(getShardSettings(r)).build();
        }

        @Override
        protected Shard mapPartial(ResultSetReader resultSet) {
            return partialShard(Shard.newBuilder(), resultSet).build();
        }

        private static Shard.Builder partialShard(Shard.Builder builder, ResultSetReader r) {
            return builder
                .setId(r.getColumn("id").getUtf8())
                .setProjectId(r.getColumn("projectId").getUtf8())
                .setFolderId(r.getColumn("folderId").getUtf8())
                .setNumId(r.getColumn("numId").getInt32())
                .setClusterId(r.getColumn("clusterId").getUtf8())
                .setServiceId(r.getColumn("serviceId").getUtf8())
                .setClusterName(r.getColumn("clusterName").getUtf8())
                .setServiceName(r.getColumn("serviceName").getUtf8())
                .setCreatedAtMillis(r.getColumn("createdAt").getInt64())
                .setUpdatedAtMillis(r.getColumn("updatedAt").getInt64())
                .setState(parseState(r.getColumn("state").getUtf8()));
        }

        private ShardSettings getShardSettings(ResultSetReader r) {
            var shardSettingsPersisted = fromJsonOrNull(objectMapper, r.getColumn("shardSettings").getUtf8(), ShardSettings.class);
            return ShardSettings.of(
                    shardSettingsPersisted != null ? shardSettingsPersisted.getType() : ShardSettings.Type.UNSPECIFIED,
                    shardSettingsPersisted != null ? shardSettingsPersisted.getPullSettings() : null,
                    shardSettingsPersisted != null ? shardSettingsPersisted.getGrid() : 0,
                    ShardSettings.getMetricsTtlDays(shardSettingsPersisted, 0),
                    DecimPolicy.parse(r.getColumn("decimPolicy").getUtf8()),
                    shardSettingsPersisted != null ? shardSettingsPersisted.getAggregationSettings() : ShardSettings.AggregationSettings.EMPTY,
                    ShardSettings.getInterval(shardSettingsPersisted, 0)
            );
        }
    }
}
