package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.db.dao.UserSettingsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.UserSettings;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonMap;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;


/**
 * @author Sergey Polovko
 */
public class YdbUserSettingsDao implements UserSettingsDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("user_settings", Arrays.asList(
        "find",
        "upsert",
        "delete"
    ));

    private final UserSettingsTable table;
    private final QueryText queryText;

    public YdbUserSettingsDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new UserSettingsTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("user.settings.table.path", tablePath));
    }

    @Override
    public CompletableFuture<List<UserSettings>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<UserSettings> findByLogin(String login) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$login", utf8(login));
            return table.queryOne(query, params)
                .thenApply(newSettings -> {
                    return newSettings.orElseGet(() -> new UserSettings(login, Collections.emptyMap()));
                });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<UserSettings> upsert(UserSettings settings) {
        try {
            String query = queryText.query("upsert");
            return table.updateOne(query, settings)
                .thenApply(ignore -> settings);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> deleteUser(String login) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$login", utf8(login));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * PROJECT MENU TABLE
     */
    private static final class UserSettingsTable extends YdbTable<String, UserSettings> {

        private final ObjectMapper objectMapper;

        UserSettingsTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("login", PrimitiveType.utf8())
                .addNullableColumn("settings", PrimitiveType.utf8())
                .setPrimaryKey("login")
                .build();
        }

        @Override
        protected String getId(UserSettings s) {
            return s.getLogin();
        }

        @Override
        protected Params toParams(UserSettings s) {
            return Params.create()
                .put("$login", utf8(s.getLogin()))
                .put("$settings", utf8(toJson(objectMapper, s.getSettings())));
        }

        @Override
        protected UserSettings mapFull(ResultSetReader r) {
            String login = r.getColumn("login").getUtf8();
            Map<String, String> settings = fromJsonMap(objectMapper, r.getColumn("settings").getUtf8());
            return new UserSettings(login, settings);
        }

        @Override
        protected UserSettings mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
