package ru.yandex.solomon.core.db.dao.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.db.dao.ViewHistoryDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.db.model.ViewHistory;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;


/**
 * @author Sergey Polovko
 */
public class YdbViewHistoryDao implements ViewHistoryDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate("view_history", Arrays.asList(
        "upsert",
        "find_for_login",
        "delete_with_login"
    ));

    private final ViewHistoryTable table;
    private final QueryText queryText;

    public YdbViewHistoryDao(TableClient tableClient, String tablePath) {
        this.table = new ViewHistoryTable(tableClient, tablePath);
        this.queryText = TEMPLATE.build(Collections.singletonMap("view.history.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Optional<ViewHistory>> upsert(ViewHistory viewHistory) {
        try {
            ViewHistory viewHistoryWithId = StringUtils.isEmpty(viewHistory.getId())
                ? viewHistory.withId(UUID.randomUUID().toString())
                : viewHistory;
            String query = queryText.query("upsert");
            return table.updateOne(query, viewHistoryWithId)
                .thenApply(ignore -> Optional.of(viewHistoryWithId));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<ViewHistory>> findAll() {
        return table.queryAll();
    }

    @Override
    public CompletableFuture<List<ViewHistory>> findHistoryByLogin(String login) {
        Params params = Params.of("$login", utf8(login));
        return table.queryPage(params, PageOptions.ALL, (opts) -> {
                return queryText.query("find_for_login", QueryText.paging(opts));
            })
            .thenApply(PagedResult::getResult);
    }

    @Override
    public CompletableFuture<Boolean> deleteById(String login, String id) {
        try {
            String query = queryText.query("delete_with_login");
            Params params = Params.of("$id", utf8(id), "$login", utf8(login));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    /**
     * VIEW HISTORY TABLE
     */
    private static final class ViewHistoryTable extends YdbTable<String, ViewHistory> {

        ViewHistoryTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("login", PrimitiveType.utf8())
                .addNullableColumn("url", PrimitiveType.utf8())
                .addNullableColumn("pageTitle", PrimitiveType.utf8())
                .addNullableColumn("pinned", PrimitiveType.bool())
                .setPrimaryKey("id")
                .build();
        }

        @Override
        protected String getId(ViewHistory h) {
            return h.getId();
        }

        @Override
        protected Params toParams(ViewHistory h) {
            return Params.create()
                .put("$id", utf8(Objects.requireNonNull(h.getId())))
                .put("$login", utf8(h.getLogin()))
                .put("$url", utf8(h.getUrl()))
                .put("$pageTitle", utf8(Nullables.orEmpty(h.getPageTitle())))
                .put("$pinned", bool(h.isPinned()));
        }

        @Override
        protected ViewHistory mapFull(ResultSetReader r) {
            String id = r.getColumn("id").getUtf8();
            String login = r.getColumn("login").getUtf8();
            String url = r.getColumn("url").getUtf8();
            String pageTitle = r.getColumn("pageTitle").getUtf8();
            boolean pinned = r.getColumn("pinned").getBool();
            return new ViewHistory(id, login, url, pageTitle, pinned);
        }

        @Override
        protected ViewHistory mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
