package ru.yandex.solomon.core.db.model;


import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Stanislav Kashirin
 */
@SuppressWarnings("AbstractClassWithoutAbstractMethods")
@ParametersAreNullableByDefault
public abstract class AbstractAuditable extends DefaultObject implements Auditable {

    protected final long createdAtMillis;
    protected final long updatedAtMillis;

    protected final String createdBy;
    protected final String updatedBy;

    protected final int version;

    protected AbstractAuditable(
        long createdAtMillis,
        long updatedAtMillis,
        String createdBy,
        String updatedBy,
        int version)
    {
        this.createdAtMillis = createdAtMillis;
        this.updatedAtMillis = updatedAtMillis;
        this.createdBy = Nullables.orEmpty(createdBy);
        this.updatedBy = Nullables.orEmpty(updatedBy);
        this.version = version;
    }

    protected AbstractAuditable(
        Instant createdAt,
        Instant updatedAt,
        String createdBy,
        String updatedBy,
        int version)
    {
        this.createdAtMillis = Nullables.orEpoch(createdAt).toEpochMilli();
        this.updatedAtMillis = Nullables.orEpoch(updatedAt).toEpochMilli();
        this.createdBy = Nullables.orEmpty(createdBy);
        this.updatedBy = Nullables.orEmpty(updatedBy);
        this.version = version;
    }

    protected AbstractAuditable(@Nonnull Auditable auditable) {
        this(
            auditable.getCreatedAtMillis(),
            auditable.getUpdatedAtMillis(),
            auditable.getCreatedBy(),
            auditable.getUpdatedBy(),
            auditable.getVersion());
    }

    @Override
    public final Instant getCreatedAt() {
        return Instant.ofEpochMilli(createdAtMillis);
    }

    @Override
    public final Instant getUpdatedAt() {
        return Instant.ofEpochMilli(updatedAtMillis);
    }

    @Override
    public final long getCreatedAtMillis() {
        return createdAtMillis;
    }

    @Override
    public final long getUpdatedAtMillis() {
        return updatedAtMillis;
    }

    @Override
    public final String getCreatedBy() {
        return createdBy;
    }

    @Override
    public final String getUpdatedBy() {
        return updatedBy;
    }

    @Override
    public final int getVersion() {
        return version;
    }
}
