package ru.yandex.solomon.core.db.model;

import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

import ru.yandex.solomon.util.collection.Nullables;

import static java.util.Collections.emptySet;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class Acl {
    private static final Interner<Acl> INTERNER = Interners.newWeakInterner();

    private static final Acl EMPTY = new Acl(emptySet(), emptySet(), emptySet(), emptySet());

    private final Set<String> canRead;
    private final Set<String> canUpdate;
    private final Set<String> canDelete;
    private final Set<String> canWrite;

    private Acl(
        Set<String> canRead,
        Set<String> canUpdate,
        Set<String> canDelete,
        Set<String> canWrite)
    {
        this.canRead = Nullables.orEmpty(canRead);
        this.canUpdate = Nullables.orEmpty(canUpdate);
        this.canDelete = Nullables.orEmpty(canDelete);
        this.canWrite = canWrite;
    }

    public static Acl of(
            @Nullable Set<String> canRead,
            @Nullable Set<String> canUpdate,
            @Nullable Set<String> canDelete,
            @Nullable Set<String> canWrite)
    {
        canRead = StringInterner.internSet(Nullables.orEmpty(canRead));
        canUpdate = StringInterner.internSet(Nullables.orEmpty(canUpdate));
        canDelete = StringInterner.internSet(Nullables.orEmpty(canDelete));
        canWrite = StringInterner.internSet(Nullables.orEmpty(canWrite));
        if (canRead.isEmpty() && canUpdate.isEmpty() && canDelete.isEmpty() && canWrite.isEmpty()) {
            return EMPTY;
        }

        return INTERNER.intern(new Acl(canRead, canUpdate, canDelete, canWrite));
    }

    public static Acl empty() {
        return EMPTY;
    }

    public Set<String> getCanRead() {
        return canRead;
    }

    public Set<String> getCanUpdate() {
        return canUpdate;
    }

    public Set<String> getCanDelete() {
        return canDelete;
    }

    public Set<String> getCanWrite() {
        return canWrite;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Acl)) {
            return false;
        }

        Acl acl = (Acl) o;

        if (!canRead.equals(acl.canRead)) {
            return false;
        }
        if (!canUpdate.equals(acl.canUpdate)) {
            return false;
        }
        if (!canDelete.equals(acl.canDelete)) {
            return false;
        }
        return canWrite.equals(acl.canWrite);
    }

    public boolean isEmpty() {
        return canRead.isEmpty() && canUpdate.isEmpty() && canDelete.isEmpty();
    }

    @Override
    public int hashCode() {
        int result = canRead.hashCode();
        result = 31 * result + canUpdate.hashCode();
        result = 31 * result + canDelete.hashCode();
        result = 31 * result + canWrite.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "Acl{" +
            "canRead=" + canRead +
            ", canUpdate=" + canUpdate +
            ", canDelete=" + canDelete +
            ", canWrite=" + canWrite +
            '}';
    }
}
