package ru.yandex.solomon.core.db.model;

import java.time.Instant;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Max Sherbakov
 */
public class Agent extends DefaultObject {
    private final String hostname;
    private final String provider;
    private final int dataPort;
    private final int managementPort;
    private final String version;
    private final Instant lastSeen;
    private final String description;
    private final int intervalSeconds;

    private Agent(Builder builder) {
        this.hostname = builder.hostname;
        this.provider = builder.provider;
        this.dataPort = builder.dataPort;
        this.managementPort = builder.managementPort;
        this.version = Nullables.orEmpty(builder.version);
        this.lastSeen = Nullables.orEpoch(builder.lastSeen);
        this.description = Nullables.orEmpty(builder.description);
        this.intervalSeconds = builder.pullIntervalSeconds;
    }

    public String getHostname() {
        return this.hostname;
    }

    public String getProvider() {
        return this.provider;
    }

    public int getDataPort() {
        return this.dataPort;
    }

    public int getManagementPort() {
        return this.managementPort;
    }

    public String getVersion() {
        return this.version;
    }

    public Instant getLastSeen() {
        return this.lastSeen;
    }

    public String getDescription() {
        return this.description;
    }

    public int getIntervalSeconds() {
        return intervalSeconds;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    /**
     * BUILDER
     */
    public static final class Builder {
        private String hostname;
        private String provider;
        private int dataPort;
        private int managementPort;
        private String version;
        private Instant lastSeen;
        private String description;
        private int pullIntervalSeconds = 15;

        protected Builder() {
        }

        protected Builder(Agent agent) {
            this.hostname = agent.getHostname();
            this.provider = agent.getProvider();
            this.dataPort = agent.getDataPort();
            this.managementPort = agent.getManagementPort();
            this.lastSeen = agent.getLastSeen();
            this.description = agent.getDescription();
            this.version = agent.getVersion();
            this.pullIntervalSeconds = agent.getIntervalSeconds();
        }

        public Builder setHostname(String hostname) {
            this.hostname = hostname;
            return this;
        }

        public Builder setProvider(String provider) {
            this.provider = provider;
            return this;
        }

        public Builder setDataPort(int port) {
            this.dataPort = port;
            return this;
        }

        public Builder setManagementPort(int port) {
            this.managementPort = port;
            return this;
        }

        public Builder setVersion(String version) {
            this.version = version;
            return this;
        }

        public Builder setLastSeen(Instant lastSeen) {
            this.lastSeen = lastSeen;
            return this;
        }

        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public Builder setPullIntervalSeconds(int pullIntervalSeconds) {
            this.pullIntervalSeconds = pullIntervalSeconds;
            return this;
        }

        public Agent build() {
            return new Agent(this);
        }
    }
}
