package ru.yandex.solomon.core.db.model;

import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author snoop
 */
@SuppressWarnings("AbstractClassWithoutAbstractMethods")
@ParametersAreNullableByDefault
public abstract class AuditableBuilder<T extends AuditableBuilder<T>> implements Auditable {

    private long createdAtMillis;
    private long updatedAtMillis;

    private String createdBy = StringUtils.EMPTY;
    private String updatedBy = StringUtils.EMPTY;

    private int version;

    protected AuditableBuilder() {
    }

    protected AuditableBuilder(@Nonnull Auditable auditable) {
        this.createdAtMillis = auditable.getCreatedAtMillis();
        this.updatedAtMillis = auditable.getUpdatedAtMillis();
        this.createdBy = auditable.getCreatedBy();
        this.updatedBy = auditable.getUpdatedBy();
        this.version = auditable.getVersion();
    }

    public final T setCreatedAt(Instant createdAt) {
        this.createdAtMillis = Nullables.orEpoch(createdAt).toEpochMilli();
        return self();
    }

    public final T setUpdatedAt(Instant updatedAt) {
        this.updatedAtMillis = Nullables.orEpoch(updatedAt).toEpochMilli();
        return self();
    }

    public final T setCreatedAtMillis(long createdAtMillis) {
        this.createdAtMillis = createdAtMillis;
        return self();
    }

    public final T setUpdatedAtMillis(long updatedAtMillis) {
        this.updatedAtMillis = updatedAtMillis;
        return self();
    }

    public final T setCreatedBy(String createdBy) {
        this.createdBy = StringInterner.intern(createdBy);
        return self();
    }

    public final T setUpdatedBy(String updatedBy) {
        this.updatedBy = StringInterner.intern(updatedBy);
        return self();
    }

    public final T setVersion(int version) {
        this.version = version;
        return self();
    }

    @Override
    public final Instant getCreatedAt() {
        return Instant.ofEpochMilli(createdAtMillis);
    }

    @Override
    public final Instant getUpdatedAt() {
        return Instant.ofEpochMilli(updatedAtMillis);
    }

    @Override
    public final long getCreatedAtMillis() {
        return createdAtMillis;
    }

    @Override
    public final long getUpdatedAtMillis() {
        return updatedAtMillis;
    }

    @Override
    public final String getCreatedBy() {
        return createdBy;
    }

    @Override
    public final String getUpdatedBy() {
        return updatedBy;
    }

    @Override
    public final int getVersion() {
        return version;
    }

    @SuppressWarnings("unchecked")
    private T self() {
        return (T) this;
    }
}
