package ru.yandex.solomon.core.db.model;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class Cluster extends AbstractAuditable {

    public static final Interner<Cluster> INTERNER = Interners.newWeakInterner();

    private final String id;
    private final String name;
    private final String projectId;
    private final String folderId;
    private final String description;

    private final List<ClusterHostListConf> hosts;
    private final List<ClusterHostUrlConf> hostUrls;
    private final List<ClusterConductorGroupConf> conductorGroups;
    private final List<ClusterConductorTagConf> conductorTags;
    private final List<ClusterNannyGroupConf> nannyGroups;
    private final List<ClusterQloudGroupConf> qloudGroups;
    private final List<ClusterNetworkConf> networks;
    private final List<ClusterYpConf> ypClusters;
    private final List<ClusterInstanceGroupConf> instanceGroups;
    private final List<ClusterCloudDnsConf> cloudDns;
    private final ShardSettings shardSettings;

    private final Map<String, String> labels;

    public Cluster(
        String id,
        String name,
        String projectId,
        String folderId,
        String description,
        List<ClusterHostListConf> hosts,
        List<ClusterHostUrlConf> hostUrls,
        List<ClusterConductorGroupConf> conductorGroups,
        List<ClusterConductorTagConf> conductorTags,
        List<ClusterNannyGroupConf> nannyGroups,
        List<ClusterQloudGroupConf> qloudGroups,
        List<ClusterNetworkConf> networks,
        List<ClusterYpConf> ypClusters,
        List<ClusterInstanceGroupConf> instanceGroups,
        List<ClusterCloudDnsConf> cloudDns,
        ShardSettings shardSettings,
        int version,
        Instant createdAt,
        Instant updatedAt,
        String createdBy,
        String updatedBy,
        Map<String, String> labels)
    {
        super(createdAt, updatedAt, createdBy, updatedBy, version);
        this.id = Objects.requireNonNull(id, "id must be not null");
        this.name = Objects.requireNonNull(name, "name must be not null");
        this.projectId = Objects.requireNonNull(projectId, "projectId must be not null");
        this.folderId = folderId;
        this.description = description;
        this.hosts = hosts;
        this.hostUrls = hostUrls;
        this.conductorGroups = conductorGroups;
        this.conductorTags = conductorTags;
        this.nannyGroups = nannyGroups;
        this.qloudGroups = qloudGroups;
        this.networks = networks;
        this.ypClusters = ypClusters;
        this.instanceGroups = instanceGroups;
        this.cloudDns = cloudDns;
        this.shardSettings = shardSettings;
        this.labels = ImmutableMap.copyOf(labels);
    }

    private Cluster(Builder builder) {
        this(
                builder.getId(),
                Nullables.orEmpty(builder.getName()),
                Nullables.orEmpty(builder.getProjectId()),
                Nullables.orEmpty(builder.getFolderId()),
                Nullables.orEmpty(builder.getDescription()),
                Nullables.orEmpty(builder.getHosts()),
                Nullables.orEmpty(builder.getHostUrls()),
                Nullables.orEmpty(builder.getConductorGroups()),
                Nullables.orEmpty(builder.getConductorTags()),
                Nullables.orEmpty(builder.getNannyGroups()),
                Nullables.orEmpty(builder.getQloudGroups()),
                Nullables.orEmpty(builder.getNetworks()),
                Nullables.orEmpty(builder.getYpClusters()),
                Nullables.orEmpty(builder.getInstanceGroups()),
                Nullables.orEmpty(builder.getCloudDns()),
                builder.getShardSettings(),
                Nullables.orZero(builder.getVersion()),
                Nullables.orEpoch(builder.getCreatedAt()),
                Nullables.orEpoch(builder.getUpdatedAt()),
                Nullables.orEmpty(builder.getCreatedBy()),
                Nullables.orEmpty(builder.getUpdatedBy()),
                Nullables.orEmpty(builder.getLabels())
        );
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    /**
     * Cluster unique id
     */
    public String getId() {
        return id;
    }

    public String getProjectId() {
        return projectId;
    }

    public String getFolderId() {
        return folderId;
    }

    public String getDescription() {
        return description;
    }

    /**
     * Cluster name, as displayed on the web
     */
    public String getName() {
        return name;
    }

    /**
     * Cluster hosts
     */
    public List<ClusterHostListConf> getHosts() {
        return hosts;
    }

    /**
     * Url of text file with list of hosts
     */
    public List<ClusterHostUrlConf> getHostUrls() {
        return hostUrls;
    }

    public List<ClusterConductorGroupConf> getConductorGroups() {
        return conductorGroups;
    }

    public List<ClusterConductorTagConf> getConductorTags() {
        return conductorTags;
    }

    public List<ClusterNannyGroupConf> getNannyGroups() {
        return nannyGroups;
    }

    public List<ClusterQloudGroupConf> getQloudGroups() {
        return qloudGroups;
    }

    public List<ClusterNetworkConf> getNetworks() {
        return networks;
    }

    public List<ClusterYpConf> getYpClusters() {
        return ypClusters;
    }

    public List<ClusterInstanceGroupConf> getInstanceGroups() {
        return instanceGroups;
    }

    public List<ClusterCloudDnsConf> getCloudDns() {
        return cloudDns;
    }

    public Map<String, String> getLabels() {
        return labels;
    }

    public ShardSettings getShardSettings() {
        return shardSettings;
    }

    public Integer getHostGroupsCount() {
        int count = 0;
        if (hosts.size() > 0) {
            count ++;
        }
        if (hostUrls.size() > 0) {
            count ++;
        }
        if (conductorGroups.size() > 0) {
            count ++;
        }
        if (conductorTags.size() > 0) {
            count ++;
        }
        if (nannyGroups.size() > 0) {
            count ++;
        }
        if (qloudGroups.size() > 0) {
            count ++;
        }
        if (networks.size() > 0) {
            count ++;
        }
        if (instanceGroups.size() > 0) {
            count ++;
        }
        if (cloudDns.size() > 0) {
            count ++;
        }
        if (ypClusters.size() > 0) {
            count ++;
        }
        return count;
    }

    public static final class Builder extends AuditableBuilder<Builder> {

        private String id;
        private String name;
        private String projectId;
        private String folderId = StringUtils.EMPTY;
        private String description = StringUtils.EMPTY;
        private List<ClusterHostListConf> hosts = Collections.emptyList();
        private List<ClusterHostUrlConf> hostUrls = Collections.emptyList();
        private List<ClusterConductorGroupConf> conductorGroups = Collections.emptyList();
        private List<ClusterConductorTagConf> conductorTags = Collections.emptyList();
        private List<ClusterNannyGroupConf> nannyGroups = Collections.emptyList();
        private List<ClusterQloudGroupConf> qloudGroups = Collections.emptyList();
        private List<ClusterNetworkConf> networks = Collections.emptyList();
        private List<ClusterYpConf> ypClusters = Collections.emptyList();
        private List<ClusterInstanceGroupConf> instanceGroups = Collections.emptyList();
        private List<ClusterCloudDnsConf> cloudDns = Collections.emptyList();
        private ShardSettings shardSettings;
        private Map<String, String> labels = ImmutableMap.of();

        private Builder() {
        }

        private Builder(Cluster cluster) {
            super(cluster);
            this.id = cluster.getId();
            this.name = cluster.getName();
            this.projectId = cluster.getProjectId();
            this.folderId = cluster.getFolderId();
            this.description = cluster.getDescription();
            this.hosts = cluster.getHosts();
            this.hostUrls = cluster.getHostUrls();
            this.conductorGroups = cluster.getConductorGroups();
            this.conductorTags = cluster.getConductorTags();
            this.nannyGroups = cluster.getNannyGroups();
            this.qloudGroups = cluster.getQloudGroups();
            this.networks = cluster.getNetworks();
            this.ypClusters = cluster.getYpClusters();
            this.instanceGroups = cluster.getInstanceGroups();
            this.cloudDns = cluster.getCloudDns();
            this.shardSettings = cluster.getShardSettings().copy();
            this.labels = ImmutableMap.copyOf(cluster.getLabels());
        }

        public Builder setId(String id) {
            this.id = StringInterner.intern(id);
            return this;
        }

        public Builder setName(String name) {
            this.name = StringInterner.intern(name);
            return this;
        }

        public Builder setProjectId(String projectId) {
            this.projectId = StringInterner.intern(projectId);
            return this;
        }

        public Builder setFolderId(String folderId) {
            this.folderId = StringInterner.intern(folderId);
            return this;
        }

        public Builder setDescription(String description) {
            this.description = StringInterner.intern(description);
            return this;
        }

        public Builder setHosts(List<ClusterHostListConf> hosts) {
            this.hosts = hosts;
            return this;
        }

        public Builder setHostUrls(List<ClusterHostUrlConf> hostUrls) {
            this.hostUrls = hostUrls;
            return this;
        }

        public Builder setConductorGroups(List<ClusterConductorGroupConf> conductorGroups) {
            this.conductorGroups = conductorGroups;
            return this;
        }

        public Builder setConductorTags(List<ClusterConductorTagConf> conductorTags) {
            this.conductorTags = conductorTags;
            return this;
        }

        public Builder setNannyGroups(List<ClusterNannyGroupConf> nannyGroups) {
            this.nannyGroups = nannyGroups;
            return this;
        }

        public Builder setQloudGroups(List<ClusterQloudGroupConf> qloudGroups) {
            this.qloudGroups = qloudGroups;
            return this;
        }

        public Builder setNetworks(List<ClusterNetworkConf> networks) {
            this.networks = networks;
            return this;
        }

        public Builder setYpClusters(List<ClusterYpConf> ypClusters) {
            this.ypClusters = ypClusters;
            return this;
        }

        public Builder setInstanceGroups(List<ClusterInstanceGroupConf> instanceGroups) {
            this.instanceGroups = instanceGroups;
            return this;
        }

        public Builder setCloudDns(List<ClusterCloudDnsConf> cloudDns) {
            this.cloudDns = cloudDns;
            return this;
        }

        public Builder setShardSettings(ShardSettings shardSettings) {
            this.shardSettings = shardSettings;
            return this;
        }

        public Builder setLabels(Map<String, String> labels) {
            var map = ImmutableMap.<String, String>builder();
            for (var entry : labels.entrySet()) {
                map.put(StringInterner.intern(entry.getKey()), StringInterner.intern(entry.getValue()));
            }
            this.labels = map.build();
            return this;
        }

        public String getId() {
            return id;
        }

        public String getName() {
            return name;
        }

        public String getProjectId() {
            return projectId;
        }

        public String getFolderId() {
            return folderId;
        }

        public String getDescription() {
            return description;
        }

        public List<ClusterHostListConf> getHosts() {
            return hosts;
        }

        public List<ClusterHostUrlConf> getHostUrls() {
            return hostUrls;
        }

        public List<ClusterConductorGroupConf> getConductorGroups() {
            return conductorGroups;
        }

        public List<ClusterConductorTagConf> getConductorTags() {
            return conductorTags;
        }

        public List<ClusterNannyGroupConf> getNannyGroups() {
            return nannyGroups;
        }

        public List<ClusterQloudGroupConf> getQloudGroups() {
            return qloudGroups;
        }

        public List<ClusterNetworkConf> getNetworks() {
            return networks;
        }

        public List<ClusterYpConf> getYpClusters() {
            return ypClusters;
        }

        public List<ClusterInstanceGroupConf> getInstanceGroups() {
            return instanceGroups;
        }

        public List<ClusterCloudDnsConf> getCloudDns() {
            return cloudDns;
        }

        public Map<String, String> getLabels() {
            return labels;
        }

        public ShardSettings getShardSettings() {
            return shardSettings;
        }

        public Cluster build() {
            return INTERNER.intern(new Cluster(this));
        }
    }
}
