package ru.yandex.solomon.core.db.model;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class ClusterNannyGroupConf extends DefaultObject {

    private static final Interner<ClusterNannyGroupConf> INTERNER = Interners.newWeakInterner();

    private final String service;
    private final String[] labels;
    private final boolean useFetchedPort;
    private final NannyEnv env;
    private final int portShift;
    private final String[] cfgGroup;

    private ClusterNannyGroupConf(
            String service,
            String[] labels,
            boolean useFetchedPort,
            NannyEnv env,
            int portShift,
            String[] cfgGroup)
    {
        this.service = StringInterner.intern(Nullables.orEmpty(service));
        this.labels = StringInterner.internArray(labels);
        this.useFetchedPort = useFetchedPort;
        this.env = env;
        this.portShift = portShift;
        this.cfgGroup = StringInterner.internArray(cfgGroup);
    }

    @JsonCreator
    public static ClusterNannyGroupConf of(
            @JsonProperty("service") String service,
            @JsonProperty("labels") String[] labels,
            @JsonProperty("useFetchedPort") Boolean useFetchedPort,
            @JsonProperty("env") NannyEnv env,
            @JsonProperty("portShift") Integer portShift,
            @JsonProperty("cfgGroup") String[] cfgGroup)
    {
        return INTERNER.intern(new ClusterNannyGroupConf(
                Nullables.orEmpty(service),
                Nullables.orEmpty(labels),
                Nullables.orFalse(useFetchedPort),
                Nullables.orDefault(env, NannyEnv.PRODUCTION),
                Nullables.orZero(portShift),
                Nullables.orEmpty(cfgGroup)));
    }
    /**
     * Name of nanny service as in /v2/services/<%service%>/current_state/instances/
     */
    @JsonProperty
    public String getService() {
        return service;
    }

    @JsonProperty
    public String[] getLabels() {
        return labels;
    }

    @JsonProperty
    public boolean isUseFetchedPort() {
        return useFetchedPort;
    }

    @JsonProperty
    public NannyEnv getEnv() {
        return env;
    }

    @JsonProperty
    public int getPortShift() {
        return portShift;
    }

    /**
     * Name of cfg group for cfg-api http://api.gencfg.yandex-team.ru/<%tag%>/groups/<%cfgGroup%>
     * <%tag%> is fetched from /v2/services/<%service%>/runtime_attrs/instances/ -> content.extended_gencfg_groups[i].release
     * @see <a href="https://wiki.yandex-team.ru/jandekspoisk/sepe/gencfg">GenCfg</a>
     */
    @JsonProperty
    public String[] getCfgGroup() {
        return cfgGroup;
    }
}
