package ru.yandex.solomon.core.db.model;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class ClusterQloudGroupConf extends DefaultObject {

    private static final Interner<ClusterQloudGroupConf> INTERNER = Interners.newWeakInterner();

    private final String component;
    private final String environment;
    private final String application;
    private final String project;
    private final String deployment;
    private final String[] labels;

    private ClusterQloudGroupConf(
            @Nullable String component,
            @Nullable String environment,
            @Nullable String application,
            @Nullable String project,
            @Nullable String deployment,
            @Nullable String[] labels)
    {
        this.component = StringInterner.intern(Nullables.orEmpty(component));
        this.environment = StringInterner.intern(Nullables.orEmpty(environment));
        this.application = StringInterner.intern(Nullables.orEmpty(application));
        this.project = StringInterner.intern(Nullables.orEmpty(project));
        this.deployment = StringInterner.intern(Nullables.orEmpty(deployment));
        this.labels = StringInterner.internArray(Nullables.orEmpty(labels));
    }

    @JsonCreator
    public static ClusterQloudGroupConf of(
        @JsonProperty("component") @Nullable String component,
        @JsonProperty("environment") @Nullable String environment,
        @JsonProperty("application") @Nullable String application,
        @JsonProperty("project") @Nullable String project,
        @JsonProperty("deployment") @Nullable String deployment,
        @JsonProperty("labels") @Nullable String[] labels)
    {
       return INTERNER.intern(new ClusterQloudGroupConf(component, environment, application, project, deployment, labels));
    }

    public static ClusterQloudGroupConf.Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder()
            .setComponent(component)
            .setEnvironment(environment)
            .setApplication(application)
            .setProject(project)
            .setDeployment(deployment)
            .setLabels(labels);
    }

    @JsonProperty
    public String getComponent() {
        return component;
    }

    @JsonProperty
    public String getEnvironment() {
        return environment;
    }

    @JsonProperty
    public String getApplication() {
        return application;
    }

    @JsonProperty
    public String getProject() {
        return project;
    }

    @JsonProperty
    public String getDeployment() {
        return deployment;
    }

    @JsonProperty
    public String[] getLabels() {
        return labels;
    }

    public static class Builder {
        @Nullable
        private String component;
        @Nullable
        private String environment;
        @Nullable
        private String application;
        @Nullable
        private String project;
        @Nullable
        private String deployment;
        @Nullable
        private String[] labels;

        public Builder setComponent(String component) {
            this.component = component;
            return this;
        }

        public Builder setEnvironment(String environment) {
            this.environment = environment;
            return this;
        }

        public Builder setApplication(String application) {
            this.application = application;
            return this;
        }

        public Builder setProject(String project) {
            this.project = project;
            return this;
        }

        public Builder setDeployment(String deployment) {
            this.deployment = deployment;
            return this;
        }

        public Builder setLabels(String[] labels) {
            this.labels = labels;
            return this;
        }

        public Builder clearComponent() {
            this.component = null;
            return this;
        }

        public Builder clearEnvironment() {
            this.environment = null;
            return this;
        }

        public Builder clearApplication() {
            this.application = null;
            return this;
        }

        public Builder clearProject() {
            this.project = null;
            return this;
        }

        public Builder clearDeployment() {
            this.deployment = null;
            return this;
        }

        public Builder clearLabels() {
            this.labels = null;
            return this;
        }

        public ClusterQloudGroupConf build() {
            return ClusterQloudGroupConf.of(
                component,
                environment,
                application,
                project,
                deployment,
                labels
            );
        }
    }
}
