package ru.yandex.solomon.core.db.model;

import java.time.Instant;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class Dashboard extends AbstractAuditable {

    private final String id;
    private final String projectId;
    private final String folderId;
    private final String name;
    private final String description;
    private final double heightMultiplier;
    private final Selector[] parameters;
    private final DashboardRow[] rows;
    private final String generatedId;

    public Dashboard(
        String id,
        String projectId,
        String folderId,
        String name,
        String description,
        double heightMultiplier,
        Selector[] parameters,
        DashboardRow[] rows,
        String generatedId,
        Instant createdAt,
        Instant updatedAt,
        String createdBy,
        String updatedBy,
        int version)
    {
        super(createdAt, updatedAt, createdBy, updatedBy, version);
        this.id = Objects.requireNonNull(id, "id must not be null");
        this.projectId = Objects.requireNonNull(projectId, "projectId must not be null");
        this.folderId = folderId;
        this.name = Objects.requireNonNull(name, "name must not be null");
        this.description = description;
        this.heightMultiplier = Math.max(1, heightMultiplier);
        this.parameters = parameters;
        this.rows = rows;
        this.generatedId = generatedId;
    }

    private Dashboard(Builder builder) {
        this(
            builder.getId(),
            builder.getProjectId(),
            builder.getFolderId(),
            builder.getName(),
            builder.getDescription(),
            builder.getHeightMultiplier(),
            builder.getParameters(),
            builder.getRows(),
            builder.getGeneratedId(),
            builder.getCreatedAt(),
            builder.getUpdatedAt(),
            builder.getCreatedBy(),
            builder.getUpdatedBy(),
            builder.getVersion()
        );
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getProjectId() {
        return projectId;
    }

    public String getFolderId() {
        return folderId;
    }

    public String getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public double getHeightMultiplier() {
        return heightMultiplier;
    }

    public Selector[] getParameters() {
        return parameters;
    }

    public DashboardRow[] getRows() {
        return rows;
    }

    public String getGeneratedId() {
        return generatedId;
    }

    public static final class Builder extends AuditableBuilder<Builder> {

        private String id;
        private String projectId;
        private String folderId = StringUtils.EMPTY;
        private String name;
        private String description = StringUtils.EMPTY;
        private double heightMultiplier = NumberUtils.DOUBLE_ONE;
        private Selector[] parameters = new Selector[0];
        private DashboardRow[] rows = new DashboardRow[0];
        private String generatedId = StringUtils.EMPTY;

        private Builder() {
        }

        private Builder(Dashboard dashboard) {
            super(dashboard);
            this.id = dashboard.getId();
            this.projectId = dashboard.getProjectId();
            this.folderId = dashboard.getFolderId();
            this.name = dashboard.getName();
            this.description = dashboard.getDescription();
            this.heightMultiplier = dashboard.getHeightMultiplier();
            this.parameters = dashboard.getParameters();
            this.rows = dashboard.getRows();
            this.generatedId = dashboard.getGeneratedId();
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public Builder setProjectId(String projectId) {
            this.projectId = projectId;
            return this;
        }

        public Builder setFolderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public Builder setHeightMultiplier(double heightMultiplier) {
            this.heightMultiplier = heightMultiplier;
            return this;
        }

        public Builder setParameters(Selector[] parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder setRows(DashboardRow[] rows) {
            this.rows = rows;
            return this;
        }

        public Builder setGeneratedId(String generatedId) {
            this.generatedId = generatedId;
            return this;
        }

        public String getId() {
            return id;
        }

        public String getProjectId() {
            return projectId;
        }

        public String getFolderId() {
            return folderId;
        }

        public String getName() {
            return name;
        }

        public String getDescription() {
            return description;
        }

        public double getHeightMultiplier() {
            return heightMultiplier;
        }

        public Selector[] getParameters() {
            return parameters;
        }

        public DashboardRow[] getRows() {
            return rows;
        }

        public String getGeneratedId() {
            return generatedId;
        }

        public Dashboard build() {
            return new Dashboard(this);
        }
    }
}
