package ru.yandex.solomon.core.db.model;

import java.time.Instant;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;

import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class Project extends AbstractAuditable {

    private static final Interner<Project> INTERNER = Interners.newWeakInterner();

    private final String id;
    private final String name;
    private final String description;
    private final String owner;
    private final Acl acl;
    private final String abcService;
    private final boolean onlyAuthRead;
    private final boolean onlyAuthPush;
    private final String metricNameLabel;
    private final boolean onlyMetricNameShards;
    private final boolean onlyNewFormatWrites;
    private final boolean onlyNewFormatReads;
    private final Map<String, String> labels;

    public Project(
        String id,
        String name,
        String description,
        String owner,
        Acl acl,
        String abcService,
        boolean onlyAuthRead,
        boolean onlyAuthPush,
        boolean onlyMetricNameShards,
        boolean onlyNewFormatWrites,
        boolean onlyNewFormatReads,
        String metricNameLabel,
        Instant createdAt,
        Instant updatedAt,
        String createdBy,
        String updatedBy,
        int version,
        Map<String, String> labels)
    {
        super(createdAt, updatedAt, createdBy, updatedBy, version);
        this.id = Objects.requireNonNull(id, "id must be not null");
        this.name = Objects.requireNonNull(name, "name must be not null");
        this.owner = Objects.requireNonNull(owner, "owner must be not null");
        this.description = description;
        this.acl = acl;
        this.abcService = abcService;
        this.onlyAuthRead = onlyAuthRead;
        this.onlyAuthPush = onlyAuthPush;
        this.onlyMetricNameShards = onlyMetricNameShards;
        this.onlyNewFormatReads = onlyNewFormatReads;
        this.onlyNewFormatWrites = onlyNewFormatWrites;
        this.metricNameLabel = metricNameLabel;
        this.labels = ImmutableMap.copyOf(Nullables.orEmpty(labels));
    }

    private Project(Builder builder) {
        super(builder);
        this.id = Objects.requireNonNull(builder.getId(), "id must be not null");
        this.name = Objects.requireNonNull(builder.getName(), "name must be not null");
        this.owner = Objects.requireNonNull(builder.getOwner(), "owner must be not null");
        this.description = Nullables.orEmpty(builder.getDescription());
        this.acl = Nullables.orDefault(builder.getAcl(), Acl.empty());
        this.abcService = Nullables.orEmpty(builder.getAbcService());
        this.onlyAuthRead = Nullables.orFalse(builder.isOnlyAuthRead());
        this.onlyAuthPush = Nullables.orFalse(builder.isOnlyAuthPush());
        this.onlyMetricNameShards = Nullables.orFalse(builder.isOnlyMetricNameShards());
        this.onlyNewFormatReads = Nullables.orFalse(builder.isOnlyNewFormatReads());
        this.onlyNewFormatWrites = Nullables.orFalse(builder.isOnlyNewFormatWrites());
        this.metricNameLabel = Nullables.orEmpty(builder.getMetricNameLabel());
        this.labels = ImmutableMap.copyOf(Nullables.orEmpty(Nullables.orEmpty(builder.getLabels())));
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public String getOwner() {
        return owner;
    }

    public Acl getAcl() {
        return acl;
    }

    public String getAbcService() {
        return abcService;
    }

    public Boolean isOnlyAuthRead() {
        return onlyAuthRead;
    }

    public boolean isOnlyAuthPush() {
        return onlyAuthPush;
    }

    public boolean isOnlyMetricNameShards() {
        return onlyMetricNameShards;
    }

    public boolean isOnlyNewFormatWrites() {
        return onlyNewFormatWrites;
    }

    public boolean isOnlyNewFormatReads() {
        return onlyNewFormatReads;
    }

    public String getMetricNameLabel() {
        return metricNameLabel;
    }

    public Map<String, String> getLabels() {
        return labels;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        Project project = (Project) o;

        if (onlyAuthRead != project.onlyAuthRead) {
            return false;
        }
        if (onlyAuthPush != project.onlyAuthPush) {
            return false;
        }
        if (onlyMetricNameShards != project.onlyMetricNameShards) {
            return false;
        }
        if (onlyNewFormatWrites != project.onlyNewFormatWrites) {
            return false;
        }
        if (onlyNewFormatReads != project.onlyNewFormatReads) {
            return false;
        }
        if (createdAtMillis != project.createdAtMillis) {
            return false;
        }
        if (updatedAtMillis != project.updatedAtMillis) {
            return false;
        }
        if (version != project.version) {
            return false;
        }
        if (!id.equals(project.id)) {
            return false;
        }
        if (!name.equals(project.name)) {
            return false;
        }
        if (!description.equals(project.description)) {
            return false;
        }
        if (!owner.equals(project.owner)) {
            return false;
        }
        if (!acl.equals(project.acl)) {
            return false;
        }
        if (!abcService.equals(project.abcService)) {
            return false;
        }
        if (!metricNameLabel.equals(project.metricNameLabel)) {
            return false;
        }
        if (!createdBy.equals(project.createdBy)) {
            return false;
        }
        if (!updatedBy.equals(project.updatedBy)) {
            return false;
        }
        return labels.equals(project.labels);
    }

    @Override
    public int hashCode() {
        int result = id.hashCode();
        result = 31 * result + name.hashCode();
        result = 31 * result + description.hashCode();
        result = 31 * result + owner.hashCode();
        result = 31 * result + acl.hashCode();
        result = 31 * result + abcService.hashCode();
        result = 31 * result + (onlyAuthRead ? 1 : 0);
        result = 31 * result + (onlyAuthPush ? 1 : 0);
        result = 31 * result + metricNameLabel.hashCode();
        result = 31 * result + (onlyMetricNameShards ? 1 : 0);
        result = 31 * result + (onlyNewFormatWrites ? 1 : 0);
        result = 31 * result + (onlyNewFormatReads ? 1 : 0);
        result = 31 * result + (int) (createdAtMillis ^ createdAtMillis >>> 32);
        result = 31 * result + (int) (updatedAtMillis ^ updatedAtMillis >>> 32);
        result = 31 * result + createdBy.hashCode();
        result = 31 * result + updatedBy.hashCode();
        result = 31 * result + version;
        result = 31 * result + labels.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    public static final class Builder extends AuditableBuilder<Builder> {

        private String id;
        private String name;
        private String description;
        private String owner;
        private Acl acl = Acl.empty();
        private String abcService = StringUtils.EMPTY;
        private Boolean onlyAuthRead;
        private Boolean onlyAuthPush;
        private Boolean onlyMetricNameShards;
        private Boolean onlyNewFormatWrites;
        private Boolean onlyNewFormatReads;
        private String metricNameLabel = StringUtils.EMPTY;
        private Map<String, String> labels = ImmutableMap.of();

        private Builder() {
        }

        private Builder(Project project) {
            super(project);
            this.id = project.getId();
            this.name = project.getName();
            this.description = project.getDescription();
            this.owner = project.getOwner();
            this.abcService = project.getAbcService();
            this.acl = project.getAcl();
            this.onlyAuthRead = project.isOnlyAuthRead();
            this.onlyAuthPush = project.isOnlyAuthPush();
            this.onlyMetricNameShards = project.isOnlyMetricNameShards();
            this.onlyNewFormatWrites = project.isOnlyNewFormatWrites();
            this.onlyNewFormatReads = project.isOnlyNewFormatReads();
            this.metricNameLabel = project.getMetricNameLabel();
            this.labels = ImmutableMap.copyOf(project.getLabels());
        }

        public String getId() {
            return id;
        }

        public Builder setId(String id) {
            this.id = StringInterner.intern(id);
            return this;
        }

        public String getName() {
            return name;
        }

        public Builder setName(String name) {
            this.name = StringInterner.intern(name);
            return this;
        }

        public String getDescription() {
            return description;
        }

        public Builder setDescription(String description) {
            this.description = StringInterner.intern(description);
            return this;
        }

        public String getOwner() {
            return owner;
        }

        public Builder setOwner(String owner) {
            this.owner = StringInterner.intern(owner);
            return this;
        }

        public Acl getAcl() {
            return acl;
        }

        public Builder setAcl(Acl acl) {
            this.acl = acl;
            return this;
        }

        public String getAbcService() {
            return abcService;
        }

        public Builder setAbcService(String abcService) {
            this.abcService = StringInterner.intern(abcService);
            return this;
        }

        public Boolean isOnlyAuthRead() {
            return onlyAuthRead;
        }

        public Builder setOnlyAuthRead(Boolean onlyAuthRead) {
            this.onlyAuthRead = onlyAuthRead;
            return this;
        }

        public Boolean isOnlyAuthPush() {
            return onlyAuthPush;
        }

        public Builder setOnlyAuthPush(Boolean onlyAuthPush) {
            this.onlyAuthPush = onlyAuthPush;
            return this;
        }

        public Boolean isOnlyMetricNameShards() {
            return onlyMetricNameShards;
        }

        public Builder setOnlyMetricNameShards(Boolean onlyMetricNameShards) {
            this.onlyMetricNameShards = onlyMetricNameShards;
            return this;
        }

        public Boolean isOnlyNewFormatWrites() {
            return onlyNewFormatWrites;
        }

        public Builder setOnlyNewFormatWrites(Boolean onlyNewFormatWrites) {
            this.onlyNewFormatWrites = onlyNewFormatWrites;
            return this;
        }

        public Boolean isOnlyNewFormatReads() {
            return onlyNewFormatReads;
        }

        public Builder setOnlyNewFormatReads(Boolean onlyNewFormatReads) {
            this.onlyNewFormatReads = onlyNewFormatReads;
            return this;
        }

        public String getMetricNameLabel() {
            return metricNameLabel;
        }

        public Builder setMetricNameLabel(String metricNameLabel) {
            this.metricNameLabel = StringInterner.intern(metricNameLabel);
            return this;
        }

        public Map<String, String> getLabels() {
            return labels;
        }

        public Builder setLabels(Map<String, String> labels) {
            var map = ImmutableMap.<String, String>builder();
            for (var entry : labels.entrySet()) {
                map.put(StringInterner.intern(entry.getKey()), StringInterner.intern(entry.getValue()));
            }
            this.labels = map.build();
            return this;
        }

        public Project build() {
            return INTERNER.intern(new Project(this));
        }
    }
}
