package ru.yandex.solomon.core.db.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class ProjectMenu extends AbstractAuditable {

    private final String id;
    private final MenuItem[] items;

    public ProjectMenu(
        String id,
        @Nullable MenuItem[] items,
        @Nullable Integer version,
        @Nullable Instant createdAt,
        @Nullable Instant updatedAt,
        @Nullable String createdBy,
        @Nullable String updatedBy)
    {
        super(createdAt, updatedAt, createdBy, updatedBy, Nullables.orZero(version));
        this.id = Objects.requireNonNull(id, "project id must not be null");
        this.items = Nullables.orDefault(items, new MenuItem[0]);
    }

    private ProjectMenu(Builder builder) {
        this(
            builder.getId(),
            builder.getItems(),
            builder.getVersion(),
            builder.getCreatedAt(),
            builder.getUpdatedAt(),
            builder.getCreatedBy(),
            builder.getUpdatedBy());
    }

    public static ProjectMenu empty(String projectId) {
        return newBuilder().setId(projectId).build();
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getId() {
        return id;
    }

    public MenuItem[] getItems() {
        return items;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        ProjectMenu that = (ProjectMenu) o;

        if (version != that.version) {
            return false;
        }
        if (createdAtMillis != that.createdAtMillis) {
            return false;
        }
        if (updatedAtMillis != that.updatedAtMillis) {
            return false;
        }
        if (!id.equals(that.id)) {
            return false;
        }
        // Probably incorrect - comparing Object[] arrays with Arrays.equals
        if (!Arrays.equals(items, that.items)) {
            return false;
        }
        if (!createdBy.equals(that.createdBy)) {
            return false;
        }
        return updatedBy.equals(that.updatedBy);
    }

    @Override
    public int hashCode() {
        int result = id.hashCode();
        result = 31 * result + Arrays.hashCode(items);
        result = 31 * result + version;
        result = 31 * result + (int) (createdAtMillis ^ createdAtMillis >>> 32);
        result = 31 * result + (int) (updatedAtMillis ^ updatedAtMillis >>> 32);
        result = 31 * result + createdBy.hashCode();
        result = 31 * result + updatedBy.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "ProjectMenu{" +
            "id='" + id + '\'' +
            ", items=" + Arrays.toString(items) +
            ", version=" + version +
            ", createdAt=" + createdAtMillis +
            ", updatedAt=" + updatedAtMillis +
            ", createdBy='" + createdBy + '\'' +
            ", updatedBy='" + updatedBy + '\'' +
            '}';
    }

    public static final class Builder extends AuditableBuilder<Builder> {

        private String id;
        private MenuItem[] items;

        private Builder() {
        }

        private Builder(ProjectMenu projectMenu) {
            super(projectMenu);
            this.id = projectMenu.getId();
            this.items = projectMenu.getItems();
        }

        public String getId() {
            return id;
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public MenuItem[] getItems() {
            return items;
        }

        public Builder setItems(MenuItem[] items) {
            this.items = items;
            return this;
        }

        public ProjectMenu build() {
            return new ProjectMenu(this);
        }
    }
}
