package ru.yandex.solomon.core.db.model;

import java.time.Instant;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.ParametersAreNullableByDefault;

import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class Quota {
    private final String namespace;
    private final String scopeType;
    @Nullable
    private final String scopeId;

    private final String indicator;
    private final long limit;

    private final Instant updatedAt;
    private final String updatedBy;

    private Quota(String namespace,
                  String scopeType,
                  @Nullable String scopeId,
                  String indicator,
                  long limit,
                  @Nullable Instant updatedAt,
                  @Nullable String updatedBy)
    {
        this.namespace = Objects.requireNonNull(namespace, "namespace must not be null");
        this.scopeType = Objects.requireNonNull(scopeType, "scopeType must not be null");
        this.scopeId = scopeId;
        this.indicator = Objects.requireNonNull(indicator, "indicator must not be null");
        this.limit = limit;

        this.updatedAt = Nullables.orEpoch(updatedAt);
        this.updatedBy = Nullables.orEmpty(updatedBy);
    }

    public boolean isDefaultScopeId() {
        return scopeId == null;
    }

    private Quota(Builder builder) {
        this(
            builder.getNamespace(),
            builder.getScopeType(),
            builder.getScopeId(),
            builder.getIndicator(),
            builder.getLimit(),
            builder.getUpdatedAt(),
            builder.getUpdatedBy()
        );
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Quota quota = (Quota) o;
        return limit == quota.limit &&
            namespace.equals(quota.namespace) &&
            scopeType.equals(quota.scopeType) &&
            Objects.equals(scopeId, quota.scopeId) &&
            indicator.equals(quota.indicator) &&
            updatedAt.equals(quota.updatedAt) &&
            updatedBy.equals(quota.updatedBy);
    }

    @Override
    public int hashCode() {
        return Objects.hash(namespace, scopeType, scopeId, indicator, limit, updatedAt, updatedBy);
    }

    @Override
    public String toString() {
        return "Quota{" +
            "namespace='" + namespace + '\'' +
            ", scopeType='" + scopeType + '\'' +
            ", scopeId='" + scopeId + '\'' +
            ", indicator='" + indicator + '\'' +
            ", limit=" + limit +
            ", updatedAt=" + updatedAt +
            ", updatedBy='" + updatedBy + '\'' +
            '}';
    }

    public Instant getUpdatedAt() {
        return updatedAt;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public String getScopeType() {
        return scopeType;
    }

    public String getNamespace() {
        return namespace;
    }

    @Nullable
    public String getScopeId() {
        return scopeId;
    }

    public String getIndicator() {
        return indicator;
    }

    public long getLimit() {
        return limit;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @ParametersAreNullableByDefault
    public static class Builder {
        private String namespace;
        private String scopeType;
        private String scopeId;
        private String indicator;
        private Long Limit;
        private Instant updatedAt;
        private String updatedBy;

        private Builder() {
        }

        protected Builder(@Nonnull Quota quota) {
            this.namespace = quota.getNamespace();
            this.scopeType = quota.getScopeType();
            this.scopeId = quota.getScopeId();
            this.indicator = quota.getIndicator();
            this.Limit = quota.getLimit();
            this.updatedAt = quota.getUpdatedAt();
            this.updatedBy = quota.getUpdatedBy();
        }

        public Builder setNamespace(@Nonnull String namespace) {
            this.namespace = namespace;
            return this;
        }

        public Builder setScopeType(@Nonnull String scopeType) {
            this.scopeType = scopeType;
            return this;
        }

        public Builder setScopeId(@Nonnull String scopeId) {
            this.scopeId = scopeId;
            return this;
        }

        public Builder setIndicator(@Nonnull String indicator) {
            this.indicator = indicator;
            return this;
        }

        public Builder setLimit(long limit) {
            this.Limit = limit;
            return this;
        }

        public Builder setUpdatedAt(@Nonnull Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public Builder setUpdatedBy(@Nonnull String updatedBy) {
            this.updatedBy = updatedBy;
            return this;
        }

        public String getNamespace() {
            return namespace;
        }

        public String getScopeType() {
            return scopeType;
        }

        public String getScopeId() {
            return scopeId;
        }

        public String getIndicator() {
            return indicator;
        }

        public long getLimit() {
            return Limit;
        }

        public Instant getUpdatedAt() {
            return updatedAt;
        }

        public String getUpdatedBy() {
            return updatedBy;
        }

        public Quota build() {
            return new Quota(this);
        }
    }
}
