package ru.yandex.solomon.core.db.model;

import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class Service extends AbstractAuditable {

    private static final Interner<Service> INTERNER = Interners.newWeakInterner();

    public static final int DEFAULT_INTERVAL = 15;
    public static final int GRID_UNKNOWN = 0;
    public static final int GRID_ABSENT = -1;

    private final String id;
    private final String projectId;
    private final String folderId;
    private final String name;
    private final String description;
    private final boolean addTsArgs;
    private final ServiceMetricConf metricConf;
    private final String metricNameLabel;

    private final ShardSettings shardSettings;

    private final String serviceProvider;

    private final Map<String, String> labels;

    private Service(Builder builder) {
        super(builder);
        this.id = Objects.requireNonNull(Strings.emptyToNull(builder.id), "id must be not null");
        this.name = Objects.requireNonNull(Strings.emptyToNull(builder.name), "name must be not null");
        this.projectId = Objects.requireNonNull(Strings.emptyToNull(builder.projectId), "projectId must be not null");
        this.folderId = Nullables.orEmpty(builder.folderId);
        this.description = Nullables.orEmpty(builder.description);
        this.addTsArgs =  builder.addTsArgs;
        this.metricConf =  Nullables.orDefault(builder.metricConf, ServiceMetricConf.empty());
        this.metricNameLabel =  Nullables.orEmpty(builder.metricNameLabel);
        this.shardSettings = builder.shardSettings;
        this.serviceProvider = Nullables.orEmpty(builder.serviceProvider);
        this.labels = ImmutableMap.copyOf(Nullables.orEmpty(builder.labels));
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    /**
     * Service id
     */
    public String getId() {
        return id;
    }

    /**
     * Service name as displayed on the web
     */
    public String getName() {
        return name;
    }

    /**
     * Service owner project id
     */
    public String getProjectId() {
        return projectId;
    }

    /**
     * Service owner folder id
     */
    public String getFolderId() {
        return folderId;
    }

    public String getDescription() {
        return description;
    }

    /**
     * Append ?now=ts&period=dur to query
     */
    public boolean isAddTsArgs() {
        return addTsArgs;
    }

    /**
     * Metric configurations
     */
    public ServiceMetricConf getMetricConf() {
        return metricConf;
    }

    /**
     * Metric name label to support old and new metric format
     */
    public String getMetricNameLabel() {
        return metricNameLabel;
    }

    public ShardSettings getShardSettings() {
        return shardSettings;
    }

    public String getServiceProvider() {
        return serviceProvider;
    }

    public Map<String, String> getLabels() {
        return labels;
    }

    public static final class Builder extends AuditableBuilder<Builder> {
        private String id;
        private String name;
        private String projectId;
        private String folderId;
        private String description;
        private boolean addTsArgs;
        private ServiceMetricConf metricConf;
        private String metricNameLabel;
        private String serviceProvider;
        private ShardSettings shardSettings;
        private Map<String, String> labels = ImmutableMap.of();

        private Builder() {
        }

        private Builder(Service service) {
            super(service);
            this.id = service.getId();
            this.projectId = service.getProjectId();
            this.folderId = service.getFolderId();
            this.name = service.getName();
            this.description = service.getDescription();
            this.addTsArgs = service.isAddTsArgs();
            this.metricConf = service.getMetricConf();
            this.metricNameLabel = service.getMetricNameLabel();
            this.serviceProvider = service.getServiceProvider();
            this.shardSettings = service.getShardSettings().copy();
            this.labels = ImmutableMap.copyOf(service.getLabels());
        }

        public Builder setId(String id) {
            this.id = StringInterner.intern(id);
            return this;
        }

        public Builder setName(String name) {
            this.name = StringInterner.intern(name);
            return this;
        }

        public Builder setProjectId(String projectId) {
            this.projectId = StringInterner.intern(projectId);
            return this;
        }

        public Builder setFolderId(String folderId) {
            this.folderId = StringInterner.intern(folderId);
            return this;
        }

        public Builder setDescription(String description) {
            this.description = StringInterner.intern(description);
            return this;
        }

        public Builder setAddTsArgs(boolean addTsArgs) {
            this.addTsArgs = addTsArgs;
            return this;
        }

        public Builder setMetricConf(ServiceMetricConf metricConf) {
            this.metricConf = metricConf;
            return this;
        }

        public Builder setMetricNameLabel(String metricNameLabel) {
            this.metricNameLabel = StringInterner.intern(metricNameLabel);
            return this;
        }

        public Builder setServiceProvider(String serviceProvider) {
            this.serviceProvider = StringInterner.intern(serviceProvider);
            return this;
        }

        public Builder setShardSettings(ShardSettings shardSettings) {
            this.shardSettings = shardSettings;
            return this;
        }

        public Builder setLabels(Map<String, String> labels) {
            var map = ImmutableMap.<String, String>builder();
            for (var entry : labels.entrySet()) {
                map.put(StringInterner.intern(entry.getKey()), StringInterner.intern(entry.getValue()));
            }
            this.labels = map.build();
            return this;
        }

        public ShardSettings getShardSettings() {
            return shardSettings;
        }

        public boolean isAddTsArgs() {
            return addTsArgs;
        }

        public ServiceMetricConf getMetricConf() {
            return metricConf;
        }

        public Service build() {
            return INTERNER.intern(new Service(this));
        }
    }
}
