package ru.yandex.solomon.core.db.model;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Splitter;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import com.google.common.collect.Lists;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Sergey Polovko
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class ServiceMetricConf extends DefaultObject {

    private static final Interner<ServiceMetricConf> INTERNER = Interners.newWeakInterner();
    private static final ServiceMetricConf EMPTY = new ServiceMetricConf(new AggrRule[0], false);

    private final AggrRule[] aggrRules;
    private final boolean rawDataMemOnly;


    @JsonCreator
    public static ServiceMetricConf of(
        @JsonProperty("aggrRules") @Nullable AggrRule[] aggrRules,
        @JsonProperty("rawDataMemOnly") @Nullable Boolean rawDataMemOnly)
    {
        return INTERNER.intern(new ServiceMetricConf(
            Nullables.orDefault(aggrRules, EMPTY.getAggrRules()),
            Nullables.orFalse(rawDataMemOnly)
        ));
    }

    private ServiceMetricConf(
        AggrRule[] aggrRules,
        boolean rawDataMemOnly)
    {
        this.aggrRules = aggrRules;
        this.rawDataMemOnly = rawDataMemOnly;
    }

    public static ServiceMetricConf empty() {
        return EMPTY;
    }

    @JsonIgnore
    public boolean isEmpty() {
        return aggrRules.length == 0 && !rawDataMemOnly;
    }

    @JsonProperty
    public AggrRule[] getAggrRules() {
        return aggrRules;
    }

    @JsonProperty
    public boolean isRawDataMemOnly() {
        return rawDataMemOnly;
    }

    /**
     * AGGREGATION RULE
     */
    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AggrRule extends DefaultObject {
        private final String[] cond;
        private final String[] target;
        private final MetricAggregation aggr;

        @JsonCreator
        public AggrRule(
                @JsonProperty("cond") String[] cond,
                @JsonProperty("target") String[] target,
                @JsonProperty("aggr") @Nullable MetricAggregation aggr)
        {
            this.cond = Nullables.orEmpty(cond);
            this.target = Nullables.orEmpty(target);
            this.aggr = aggr;
        }

        private static String[] split(String str) {
            var it = Splitter.on(",").trimResults().omitEmptyStrings().split(str).iterator();
            return Lists.newArrayList(it).toArray(new String[0]);
        }

        public static AggrRule of(String cond, String target, @Nullable MetricAggregation aggr) {
            return new AggrRule(split(cond), split(target), aggr);
        }

        @JsonProperty
        public String[] getCond() {
            return cond;
        }

        @JsonProperty
        public String[] getTarget() {
            return target;
        }

        @Nullable
        @JsonProperty("aggr")
        public MetricAggregation getAggr() {
            return aggr;
        }
    }
}
