package ru.yandex.solomon.core.db.model;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import it.unimi.dsi.fastutil.ints.IntArrayList;

import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@ParametersAreNonnullByDefault
public final class ServiceProvider extends AbstractAuditable {
    private final String id;
    private final String description;
    private final ServiceProviderShardSettings shardSettings;
    private final List<ReferenceConf> references;
    private final String abcService;
    private final String cloudId;
    private final String tvmDestId;
    private final boolean hasGlobalId;
    private final List<String> iamServiceAccountIds;
    private final List<Integer> tvmServiceIds;

    private ServiceProvider(Builder builder) {
        super(builder);
        this.id = Objects.requireNonNull(builder.id, "id must not be null");
        this.description = Nullables.orEmpty(builder.description);
        this.shardSettings = Nullables.orDefault(builder.shardSettings, ServiceProviderShardSettings.EMPTY);
        this.references = Nullables.orEmpty(builder.references);
        this.abcService = Nullables.orEmpty(builder.abcService);
        this.cloudId = Nullables.orEmpty(builder.cloudId);
        this.tvmDestId = Nullables.orEmpty(builder.tvmDestId);
        this.iamServiceAccountIds = Nullables.orEmpty(builder.iamServiceAccountIds);
        this.tvmServiceIds = Nullables.orEmpty(builder.tvmServiceIds);
        this.hasGlobalId = builder.hasGlobalId;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getId() {
        return id;
    }

    public String getDescription() {
        return description;
    }

    public ServiceProviderShardSettings getShardSettings() {
        return shardSettings;
    }

    public String getAbcService() {
        return abcService;
    }

    public String getCloudId() {
        return cloudId;
    }

    public String getTvmDestId() {
        return tvmDestId;
    }

    public List<String> getIamServiceAccountIds() {
        return iamServiceAccountIds;
    }

    public List<Integer> getTvmServiceIds() {
        return tvmServiceIds;
    }

    public List<ReferenceConf> getReferences() {
        return references;
    }

    public boolean isHasGlobalId() {
        return hasGlobalId;
    }

    public static final class Builder extends AuditableBuilder<Builder> {
        private String id;
        private String description;
        private ServiceProviderShardSettings shardSettings;
        private List<ReferenceConf> references;
        private String abcService;
        private String cloudId;
        private String tvmDestId;
        private List<String> iamServiceAccountIds;
        private List<Integer> tvmServiceIds;
        private boolean hasGlobalId;

        private Builder() {
        }

        private Builder(@Nonnull ServiceProvider serviceProvider) {
            super(serviceProvider);
            this.id = serviceProvider.id;
            this.description = serviceProvider.description;
            this.shardSettings = serviceProvider.shardSettings;
            this.references = serviceProvider.references;
            this.abcService = serviceProvider.abcService;
            this.cloudId = serviceProvider.cloudId;
            this.tvmDestId = serviceProvider.tvmDestId;
            this.iamServiceAccountIds = serviceProvider.iamServiceAccountIds;
            this.tvmServiceIds = serviceProvider.tvmServiceIds;
            this.hasGlobalId = serviceProvider.hasGlobalId;
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public Builder setHasGlobalId(boolean hasGlobalId) {
            this.hasGlobalId = hasGlobalId;
            return this;
        }

        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public Builder setShardSettings(@Nullable ServiceProviderShardSettings shardSettings) {
            this.shardSettings = shardSettings;
            return this;
        }

        public Builder setReferences(List<ReferenceConf> references) {
            this.references = references;
            return this;
        }

        public Builder setAbcService(String abcService) {
            this.abcService = abcService;
            return this;
        }

        public Builder setCloudId(String cloudId) {
            this.cloudId = cloudId;
            return this;
        }

        public Builder setTvmDestId(String tvmDestId) {
            this.tvmDestId = tvmDestId;
            return this;
        }

        public Builder setIamServiceAccountIds(List<String> iamServiceAccountIds) {
            this.iamServiceAccountIds = iamServiceAccountIds;
            return this;
        }

        public Builder setTvmServiceIds(IntArrayList tvmServiceIds) {
            this.tvmServiceIds = tvmServiceIds;
            return this;
        }

        public ServiceProvider build() {
            return new ServiceProvider(this);
        }
    }
}
