package ru.yandex.solomon.core.db.model;

import java.time.Instant;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class Shard extends AbstractAuditable {

    public static final Interner<Shard> INTERNER = Interners.newWeakInterner();

    public static final int DEFAULT_PER_URL_QUOTA = 10_000;
    public static final int DEFAULT_FILE_METRIC_QUOTA = 1_000_000;
    public static final int DEFAULT_MEM_METRIC_QUOTA = 500_000;
    public static final int DEFAULT_RESPONSE_SIZE_QUOTA = 10 << 20; // 10 MB
    public static final int DEFAULT_PARTITIONS = 1;

    public static final int PARTITIONS_HARD_LIMIT = 100;

    private final String id;
    private final int numId;
    private final String projectId;
    private final String folderId;
    private final String clusterId;
    private final String serviceId;
    private final String description;

    // We must add this fields to support unique shards by cluster and service name, see SOLOMON-2650
    private final String clusterName;
    private final String serviceName;

    // TODO: separate next fields into sub object (ShardQuotas)
    //       do not forget to update ShardsDao.partialUpdate()
    private final int maxMetricsPerUrl;
    private final int maxFileMetrics;
    private final int maxMemMetrics;
    private final int maxResponseSizeBytes;
    private final ValidationMode validationMode;
    private final String metricNameLabel;
    private final int numPartitions;

    private final ShardSettings shardSettings;

    private final ShardState state;

    private final Map<String, String> labels;

    public Shard(
        String id,
        int numId,
        String projectId,
        @Nullable String folderId,
        String clusterId,
        String serviceId,
        String clusterName,
        String serviceName,
        @Nullable String description,
        @Nullable Integer maxMetricsPerUrl,
        @Nullable Integer maxFileMetrics,
        @Nullable Integer maxMemMetrics,
        @Nullable Integer maxResponseSizeBytes,
        @Nullable Integer numPartitions,
        @Nullable ValidationMode validationMode,
        @Nullable String metricNameLabel,
        ShardSettings shardSettings,
        @Nullable ShardState state,
        @Nullable Integer version,
        @Nullable Instant createdAt,
        @Nullable Instant updatedAt,
        @Nullable String createdBy,
        @Nullable String updatedBy,
        @Nullable Map<String, String> labels)
    {
        this(
            id,
            numId,
            projectId,
            Nullables.orEmpty(folderId),
            clusterId,
            serviceId,
            clusterName,
            serviceName,
            Nullables.orEmpty(description),
            Nullables.orDefault(maxMetricsPerUrl, DEFAULT_PER_URL_QUOTA),
            Nullables.orDefault(maxFileMetrics, DEFAULT_FILE_METRIC_QUOTA),
            Nullables.orDefault(maxMemMetrics, DEFAULT_MEM_METRIC_QUOTA),
            Nullables.orDefault(maxResponseSizeBytes, DEFAULT_RESPONSE_SIZE_QUOTA),
            Nullables.orDefault(numPartitions, DEFAULT_PARTITIONS),
            Nullables.orDefault(validationMode, ValidationMode.LEGACY_SKIP),
            Nullables.orEmpty(metricNameLabel),
            shardSettings,
            Nullables.orDefault(state, ShardState.DEFAULT),
            Nullables.orZero(version),
            Nullables.orEpoch(createdAt),
            Nullables.orEpoch(updatedAt),
            Nullables.orEmpty(createdBy),
            Nullables.orEmpty(updatedBy),
            Nullables.orEmpty(labels)
        );
    }

    public Shard(
        String id,
        int numId,
        String projectId,
        String folderId,
        String clusterId,
        String serviceId,
        String clusterName,
        String serviceName,
        String description,
        int maxMetricsPerUrl,
        int maxFileMetrics,
        int maxMemMetrics,
        int maxResponseSizeBytes,
        int numPartitions,
        ValidationMode validationMode,
        String metricNameLabel,
        ShardSettings shardSettings,
        ShardState state,
        int version,
        Instant createdAt,
        Instant updatedAt,
        String createdBy,
        String updatedBy,
        Map<String, String> labels)
    {
        super(createdAt, updatedAt, createdBy, updatedBy, version);
        this.id = Objects.requireNonNull(id, "id must be not null");
        this.numId = numId;
        this.projectId = Objects.requireNonNull(projectId, "projectId must not be null");
        this.folderId = folderId;
        this.clusterId = Objects.requireNonNull(clusterId, "clusterId must not be null");
        this.serviceId = Objects.requireNonNull(serviceId, "serviceId must not be null");
        this.clusterName = Objects.requireNonNull(clusterName, "clusterName must not be null");
        this.serviceName = Objects.requireNonNull(serviceName, "serviceName must not be null");
        this.description = description;
        this.maxMetricsPerUrl = maxMetricsPerUrl;
        this.maxFileMetrics = maxFileMetrics;
        this.maxMemMetrics = maxMemMetrics;
        this.maxResponseSizeBytes = maxResponseSizeBytes;
        this.numPartitions = numPartitions;
        this.validationMode = validationMode;
        this.metricNameLabel = metricNameLabel;
        this.shardSettings = shardSettings;
        this.state = state;
        this.labels = ImmutableMap.copyOf(labels);
    }

    private Shard(Builder builder) {
        // It's need to call nessesary constructor with default values
        // TODO (rorewillo@): use single data constructor without default value initialization
        this(
            builder.getId(),
            builder.getNumId(),
            builder.getProjectId(),
            builder.getFolderId(),
            builder.getClusterId(),
            builder.getServiceId(),
            builder.getClusterName(),
            builder.getServiceName(),
            builder.getDescription(),
            Integer.valueOf(builder.getMaxMetricsPerUrl()),
            Integer.valueOf(builder.getMaxFileMetrics()),
            Integer.valueOf(builder.getMaxMemMetrics()),
            Integer.valueOf(builder.getMaxResponseSizeBytes()),
            Integer.valueOf(builder.getNumPartitions()),
            builder.getValidationMode(),
            builder.getMetricNameLabel(),
            builder.getShardSettings(),
            builder.getState(),
            Integer.valueOf(builder.getVersion()),
            builder.getCreatedAt(),
            builder.getUpdatedAt(),
            builder.getCreatedBy(),
            builder.getUpdatedBy(),
            builder.getLabels()
        );
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    /**
     * Shard unique id
     */
    public String getId() {
        return id;
    }

    public int getNumId() {
        return numId;
    }

    /**
     * Shard owner project id
     */
    public String getProjectId() {
        return projectId;
    }

    public String getFolderId() {
        return folderId;
    }

    public String getClusterId() {
        return clusterId;
    }

    public String getServiceId() {
        return serviceId;
    }

    public String getClusterName() {
        return clusterName;
    }

    public String getServiceName() {
        return serviceName;
    }

    public String getDescription() {
        return description;
    }

    public int getMaxMetricsPerUrl() {
        return maxMetricsPerUrl;
    }

    public int getMaxFileMetrics() {
        return maxFileMetrics;
    }

    public int getMaxMemMetrics() {
        return maxMemMetrics;
    }

    public int getMaxResponseSizeBytes() {
        return maxResponseSizeBytes;
    }

    public int getNumPartitions() {
        return numPartitions;
    }

    public ShardState getState() {
        return state;
    }

    public ValidationMode getValidationMode() {
        return validationMode;
    }

    public String getMetricNameLabel() {
        return metricNameLabel;
    }

    public Map<String, String> getLabels() {
        return labels;
    }

    public ShardSettings getShardSettings() {
        return shardSettings;
    }

    public boolean hasAnyNonDefaultQuota() {
        return maxMetricsPerUrl != DEFAULT_PER_URL_QUOTA ||
            maxFileMetrics != DEFAULT_FILE_METRIC_QUOTA ||
            maxMemMetrics != DEFAULT_MEM_METRIC_QUOTA ||
            maxResponseSizeBytes != DEFAULT_RESPONSE_SIZE_QUOTA ||
            numPartitions != DEFAULT_PARTITIONS;
    }

    public static final class Builder extends AuditableBuilder<Builder> {

        private String id;
        private int numId;
        private String projectId;
        private String folderId = StringUtils.EMPTY;
        private String clusterId;
        private String serviceId;
        private String clusterName;
        private String serviceName;
        private String description;
        private int maxMetricsPerUrl = DEFAULT_PER_URL_QUOTA;
        private int maxFileMetrics = DEFAULT_FILE_METRIC_QUOTA;
        private int maxMemMetrics = DEFAULT_MEM_METRIC_QUOTA;
        private int maxResponseSizeBytes = DEFAULT_RESPONSE_SIZE_QUOTA;
        private int numPartitions = DEFAULT_PARTITIONS;
        private ValidationMode validationMode = ValidationMode.LEGACY_SKIP;
        private String metricNameLabel;
        private ShardState state = ShardState.DEFAULT;
        private ShardSettings shardSettings;
        private Map<String, String> labels = ImmutableMap.of();

        private Builder() {
        }

        private Builder(Shard shard) {
            super(shard);
            this.id = shard.getId();
            this.numId = shard.numId;
            this.projectId = shard.getProjectId();
            this.folderId = shard.getFolderId();
            this.clusterId = shard.getClusterId();
            this.serviceId = shard.getServiceId();
            this.clusterName = shard.getClusterName();
            this.serviceName = shard.getServiceName();
            this.description = shard.getDescription();
            this.maxMetricsPerUrl = shard.getMaxMetricsPerUrl();
            this.maxFileMetrics = shard.getMaxFileMetrics();
            this.maxMemMetrics = shard.getMaxMemMetrics();
            this.maxResponseSizeBytes = shard.getMaxResponseSizeBytes();
            this.numPartitions = shard.getNumPartitions();
            this.state = shard.getState();
            this.validationMode = shard.getValidationMode();
            this.metricNameLabel = shard.getMetricNameLabel();
            this.shardSettings = shard.getShardSettings() != null ? shard.getShardSettings().copy() : null;
            this.labels = ImmutableMap.copyOf(shard.getLabels());
        }

        public Builder setId(String id) {
            this.id = StringInterner.intern(id);
            return this;
        }

        public Builder setNumId(int numId) {
            this.numId = numId;
            return this;
        }

        public Builder setProjectId(String projectId) {
            this.projectId = StringInterner.intern(projectId);
            return this;
        }

        public Builder setFolderId(String folderId) {
            this.folderId = StringInterner.intern(folderId);
            return this;
        }

        public Builder setClusterId(String clusterId) {
            this.clusterId = StringInterner.intern(clusterId);
            return this;
        }

        public Builder setServiceId(String serviceId) {
            this.serviceId = StringInterner.intern(serviceId);
            return this;
        }

        public Builder setClusterName(String clusterName) {
            this.clusterName = StringInterner.intern(clusterName);
            return this;
        }

        public Builder setServiceName(String serviceName) {
            this.serviceName = StringInterner.intern(serviceName);
            return this;
        }

        public Builder setDescription(String description) {
            this.description = StringInterner.intern(description);
            return this;
        }

        public Builder setMaxMetricsPerUrl(int maxMetricsPerUrl) {
            this.maxMetricsPerUrl = maxMetricsPerUrl;
            return this;
        }

        public Builder setMaxFileMetrics(int maxFileMetrics) {
            this.maxFileMetrics = maxFileMetrics;
            return this;
        }

        public Builder setMaxMemMetrics(int maxMemMetrics) {
            this.maxMemMetrics = maxMemMetrics;
            return this;
        }

        public Builder setMaxResponseSizeBytes(int maxResponseSizeBytes) {
            this.maxResponseSizeBytes = maxResponseSizeBytes;
            return this;
        }

        public Builder setNumPartitions(int numPartitions) {
            this.numPartitions = numPartitions;
            return this;
        }

        public Builder setValidationMode(ValidationMode validationMode) {
            this.validationMode = validationMode;
            return this;
        }

        public Builder setMetricNameLabel(String metricNameLabel) {
            this.metricNameLabel = StringInterner.intern(metricNameLabel);
            return this;
        }

        public Builder setState(ShardState state) {
            this.state = state;
            return this;
        }

        public Builder setShardSettings(ShardSettings shardSettings) {
            this.shardSettings = shardSettings;
            return this;
        }

        public Builder setLabels(Map<String, String> labels) {
            var map = ImmutableMap.<String, String>builder();
            for (var entry : labels.entrySet()) {
                map.put(StringInterner.intern(entry.getKey()), StringInterner.intern(entry.getValue()));
            }
            this.labels = map.build();
            return this;
        }

        public String getId() {
            return id;
        }

        public int getNumId() {
            return numId;
        }

        public String getProjectId() {
            return projectId;
        }

        public String getFolderId() {
            return folderId;
        }

        public String getClusterId() {
            return clusterId;
        }

        public String getServiceId() {
            return serviceId;
        }

        public String getClusterName() {
            return clusterName;
        }

        public String getServiceName() {
            return serviceName;
        }

        public String getDescription() {
            return description;
        }

        public int getMaxMetricsPerUrl() {
            return maxMetricsPerUrl;
        }

        public int getMaxFileMetrics() {
            return maxFileMetrics;
        }

        public int getMaxMemMetrics() {
            return maxMemMetrics;
        }

        public int getMaxResponseSizeBytes() {
            return maxResponseSizeBytes;
        }

        public int getNumPartitions() {
            return numPartitions;
        }

        public ValidationMode getValidationMode() {
            return validationMode;
        }

        public String getMetricNameLabel() {
            return metricNameLabel;
        }

        public ShardState getState() {
            return state;
        }

        public ShardSettings getShardSettings() {
            return shardSettings;
        }

        public Map<String, String> getLabels() {
            return labels;
        }

        public Shard build() {
            return INTERNER.intern(new Shard(this));
        }
    }
}
