package ru.yandex.solomon.core.db.model;

import java.util.Arrays;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.core.conf.PushOrPull;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@ParametersAreNonnullByDefault
public class ShardSettings extends DefaultObject {
    private static final Interner<ShardSettings> INTERNER = Interners.newWeakInterner();

    private final Type type;
    @Nullable
    private final PullSettings pullSettings;
    private final int grid;
    private final int interval;
    private final int metricsTtl;
    private final DecimPolicy retentionPolicy;
    private final AggregationSettings aggregationSettings;

    private ShardSettings(
            Type type,
            @Nullable PullSettings pullSettings,
            int grid,
            int metricsTtl,
            DecimPolicy retentionPolicy,
            AggregationSettings aggregationSettings,
            @Nullable Integer interval) {
        this.type = type;
        this.pullSettings = pullSettings;
        this.grid = grid;
        this.metricsTtl = metricsTtl;
        this.retentionPolicy = retentionPolicy;
        this.aggregationSettings = aggregationSettings;
        this.interval = Nullables.orZero(interval);
    }

    @JsonCreator
    public static ShardSettings of(
            @JsonProperty("type") Type type,
            @Nullable @JsonProperty("pullSettings") PullSettings pullSettings,
            @JsonProperty("grid") int grid,
            @JsonProperty("metricsTtl") int metricsTtl,
            @JsonProperty("retentionPolicy") DecimPolicy retentionPolicy,
            @JsonProperty("aggregationSettings") AggregationSettings aggregationSettings,
            @Nullable @JsonProperty("interval") Integer interval) {
        return INTERNER.intern(new ShardSettings(
                type,
                pullSettings,
                grid,
                metricsTtl,
                retentionPolicy,
                aggregationSettings,
                interval
        ));
    }

    public static ServiceMetricConf toServiceMetricConf(@Nullable ShardSettings shardSettings, @Nullable ServiceMetricConf defaultConf) {
        if (shardSettings != null && !shardSettings.getAggregationSettings().equals(AggregationSettings.EMPTY)) {
            return ServiceMetricConf.of(shardSettings.getAggregationSettings().getAggrRules(), shardSettings.getAggregationSettings().isMemOnly());
        }
        return defaultConf;
    }

    public static Integer getMetricsTtlDays(@Nullable ShardSettings shardSettings, @Nullable Integer defaultValue) {
        if (shardSettings != null && shardSettings.getMetricsTtl() > 0) {
            return shardSettings.getMetricsTtl();
        }
        return defaultValue;
    }

    public static Integer getGrid(@Nullable ShardSettings shardSettings, @Nullable Integer defaultValue) {
        if (shardSettings != null && shardSettings.getGrid() > 0) {
            return shardSettings.getGrid();
        }

        return defaultValue;
    }

    public static DecimPolicy getDecimPolicy(@Nullable ShardSettings shardSettings, DecimPolicy defaultValue) {
        if (shardSettings != null && shardSettings.getRetentionPolicy() != null) {
            return shardSettings.getRetentionPolicy();
        }
        return defaultValue;
    }

    public static Integer getInterval(@Nullable ShardSettings shardSettings, @Nullable Integer defaultValue) {
        if (shardSettings != null && shardSettings.getInterval() > 0) {
            return shardSettings.getInterval();
        }
        return defaultValue;
    }

    public static PushOrPull getPushOrPull(@Nullable ShardSettings shardSettings, @Nullable PushOrPull defaultValue) {
        if (shardSettings != null && shardSettings.getType() == ShardSettings.Type.PULL) {
            return PushOrPull.PULL;
        } else if (shardSettings != null && shardSettings.getType() == ShardSettings.Type.PUSH) {
            return PushOrPull.PUSH;
        }
        return defaultValue;
    }

    public static int getPort(@Nullable ShardSettings shardSettings, int defaultValue) {
        if (shardSettings != null && shardSettings.getPullSettings() != null) {
            return shardSettings.getPullSettings().getPort();
        }
        return defaultValue;
    }

    public static String getPath(@Nullable ShardSettings shardSettings, @Nullable String defaultValue) {
        if (shardSettings != null && shardSettings.getPullSettings() != null) {
            return shardSettings.getPullSettings().getPath();
        }
        return defaultValue;
    }

    public static PullProtocol getProtocol(@Nullable ShardSettings shardSettings, @Nonnull PullProtocol defaultValue) {
        if (shardSettings != null && shardSettings.getPullSettings() != null) {
            return shardSettings.getPullSettings().getProtocol();
        }
        return defaultValue;
    }

    public static boolean getAddTsArgs(@Nullable ShardSettings shardSettings, boolean defaultValue) {
        if (shardSettings != null && shardSettings.getPullSettings() != null) {
            return shardSettings.getPullSettings().isAddTsArgs();
        }
        return defaultValue;
    }

    public static String getTvmDestinationId(@Nullable ShardSettings shardSettings, @Nonnull String defaultValue) {
        if (shardSettings != null && shardSettings.getPullSettings() != null) {
            return shardSettings.getPullSettings().getTvmDestinationId();
        }
        return defaultValue;
    }

    public static HostLabelPolicy getHostLabelPolicy(@Nullable ShardSettings shardSettings, @Nonnull HostLabelPolicy defaultValue) {
        if (shardSettings != null && shardSettings.getPullSettings() != null) {
            return shardSettings.getPullSettings().getHostLabelPolicy();
        }
        return defaultValue;
    }

    @JsonProperty
    public Type getType() {
        return type;
    }

    @Nullable
    @JsonProperty
    public PullSettings getPullSettings() {
        return pullSettings;
    }

    @JsonProperty
    public int getGrid() {
        return grid;
    }

    @JsonProperty
    public int getMetricsTtl() {
        return metricsTtl;
    }

    @JsonProperty
    public DecimPolicy getRetentionPolicy() {
        return retentionPolicy;
    }

    @JsonProperty
    public AggregationSettings getAggregationSettings() {
        return aggregationSettings;
    }

    @JsonProperty
    public int getInterval() {
        return interval;
    }

    public ShardSettings copy() {
        return new ShardSettings(
                type,
                pullSettings == null ? null : pullSettings.copy(),
                grid,
                metricsTtl,
                retentionPolicy,
                aggregationSettings.copy(),
                interval
        );
    }

    public enum Type {
        PULL,
        PUSH,
        @JsonEnumDefaultValue
        UNSPECIFIED,
    }

    public enum PullProtocol {
        @JsonEnumDefaultValue
        UNKNOWN,
        HTTP,
        HTTPS,
    }

    public enum HostLabelPolicy {
        @JsonEnumDefaultValue
        UNKNOWN,
        SHORT_HOSTNAME,
        FULL_HOSTNAME,
        NO_HOSTNAME,
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AggregationSettings extends DefaultObject {
        private static final Interner<AggregationSettings> INTERNER = Interners.newWeakInterner();
        public static final AggregationSettings EMPTY = of(false, new AggrRule[0], false);

        private final boolean enabled;
        private final ServiceMetricConf.AggrRule[] aggrRules;
        private final boolean memOnly;

        private AggregationSettings(boolean enabled, ServiceMetricConf.AggrRule[] aggrRules, boolean memOnly) {
            this.enabled = enabled;
            this.aggrRules = aggrRules;
            this.memOnly = memOnly;
        }

        @JsonCreator
        public static AggregationSettings of(
                @JsonProperty("enabled") boolean enabled,
                @JsonProperty("aggrRules") ServiceMetricConf.AggrRule[] aggrRules,
                @JsonProperty("memOnly") boolean memOnly)
        {
            return INTERNER.intern(new AggregationSettings(enabled, aggrRules, memOnly));
        }

        @JsonProperty
        public boolean isEnabled() {
            return enabled;
        }

        @JsonProperty
        public ServiceMetricConf.AggrRule[] getAggrRules() {
            return aggrRules;
        }

        @JsonProperty
        public boolean isMemOnly() {
            return memOnly;
        }

        public AggregationSettings copy() {
            if (this.equals(EMPTY)) {
                return EMPTY;
            }
            return AggregationSettings.of(enabled, copy(aggrRules), memOnly);
        }

        private ServiceMetricConf.AggrRule[] copy(ServiceMetricConf.AggrRule[] aggrRules) {
            ServiceMetricConf.AggrRule[] r = new ServiceMetricConf.AggrRule[aggrRules.length];
            for (int i = 0; i < aggrRules.length; i++) {
                ServiceMetricConf.AggrRule rule = new ServiceMetricConf.AggrRule(
                        Arrays.copyOf(aggrRules[i].getCond(), aggrRules[i].getCond().length),
                        Arrays.copyOf(aggrRules[i].getTarget(), aggrRules[i].getTarget().length),
                        aggrRules[i].getAggr());
                r[i] = rule;
            }
            return r;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class PullSettings extends DefaultObject {
        private static Interner<PullSettings> INTERNER = Interners.newWeakInterner();

        private final int port;
        private final String path;
        private final boolean addTsArgs;
        private final PullProtocol protocol;
        private final String tvmDestinationId;
        private final HostLabelPolicy hostLabelPolicy;

        @JsonCreator
        public static PullSettings of(
                @JsonProperty("port") int port,
                @JsonProperty("path") String path,
                @JsonProperty("addTsArgs") boolean addTsArgs,
                @JsonProperty("protocol") PullProtocol protocol,
                @JsonProperty("tvmDestinationId") String tvmDestinationId,
                @JsonProperty("hostLabelPolicy") HostLabelPolicy hostLabelPolicy)
        {
            return newBuilder()
                    .setPort(port)
                    .setPath(path)
                    .setAddTsArgs(addTsArgs)
                    .setProtocol(protocol)
                    .setTvmDestinationId(tvmDestinationId)
                    .setHostLabelPolicy(hostLabelPolicy)
                    .build();
        }

        private PullSettings(Builder builder) {
            this.port = builder.port;
            this.path = builder.path;
            this.addTsArgs = builder.addTsArgs;
            this.protocol = builder.protocol;
            this.tvmDestinationId = builder.tvmDestinationId;
            this.hostLabelPolicy = builder.hostLabelPolicy;
        }

        public static PullSettings.Builder newBuilder() {
            return new PullSettings.Builder();
        }

        @JsonProperty
        public int getPort() {
            return port;
        }

        @JsonProperty
        public String getPath() {
            return path;
        }

        @JsonProperty
        public boolean isAddTsArgs() {
            return addTsArgs;
        }

        @JsonProperty
        public PullProtocol getProtocol() {
            return protocol;
        }

        @JsonProperty
        public String getTvmDestinationId() {
            return tvmDestinationId;
        }

        @JsonProperty
        public HostLabelPolicy getHostLabelPolicy() {
            return hostLabelPolicy;
        }

        public PullSettings copy() {
            return newBuilder()
                    .setPort(port)
                    .setPath(path)
                    .setAddTsArgs(addTsArgs)
                    .setProtocol(protocol)
                    .setTvmDestinationId(tvmDestinationId)
                    .setHostLabelPolicy(hostLabelPolicy)
                    .build();
        }

        public static class Builder {
            private int port = 0;
            private String path = "";
            private boolean addTsArgs = false;
            private PullProtocol protocol = PullProtocol.UNKNOWN;
            private String tvmDestinationId = "";
            private HostLabelPolicy hostLabelPolicy = HostLabelPolicy.UNKNOWN;

            public Builder setPort(int port) {
                this.port = port;
                return this;
            }

            public Builder setPath(String path) {
                this.path = StringInterner.intern(path);
                return this;
            }

            public Builder setAddTsArgs(boolean addTsArgs) {
                this.addTsArgs = addTsArgs;
                return this;
            }

            public Builder setProtocol(PullProtocol protocol) {
                this.protocol = protocol;
                return this;
            }

            public Builder setTvmDestinationId(String tvmDestinationId) {
                this.tvmDestinationId = StringInterner.intern(tvmDestinationId);
                return this;
            }

            public Builder setHostLabelPolicy(HostLabelPolicy hostLabelPolicy) {
                this.hostLabelPolicy = hostLabelPolicy;
                return this;
            }

            public PullSettings build() {
                return INTERNER.intern(new PullSettings(this));
            }
        }
    }
}
